<?php

declare(strict_types=1);

use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;
use Points\Core\Extension\QuoteAddress;

class Points_Core_Helper_Data extends Mage_Core_Helper_Data {
    const CONFIG_PROVIDERS = "points/providers";
    const SALES_POINTS_TYPE_KEY = "poionts_type";

    /**
     * Returns a list of *all* type providers in the given store, with the type
     * id as key.
     *
     * @return Array<string, ProviderInterface>
     */
    public function getTypeProviders(Mage_Core_Model_Store $store): array {
        /**
         * @var Array<string, array{model?: ?string}>
         */
        $data = $store->getConfig(self::CONFIG_PROVIDERS) ?: [];

        $items = array_combine(array_keys($data), array_map(function($item, $key) {
            $model = Mage::getSingleton($item["model"] ?? "");

            if( ! $model instanceof ProviderInterface) {
                throw new Exception(sprintf(
                    "%s: Model \"%s\" defined in %s/%s/model does not implement %s",
                    __CLASS__,
                    $item["model"] ?? "",
                    self::CONFIG_PROVIDERS,
                    $key,
                    ProviderInterface::class
                ));
            }

            return $model;
        }, $data, array_keys($data)));

        return $items;
    }

    /**
     * Returns a list of enabled type providers in the given store, with the type
     * id as key.
     *
     * @return Array<string, ProviderInterface>
     */
    public function getEnabledTypeProviders(Mage_Core_Model_Store $store): array {
        return array_filter(
            $this->getTypeProviders($store),
            function($p) use($store): bool {
                return $p->isEnabled($store);
            }
        );
    }

    /**
     * Returns a list of type providers which apply to the quote, with the type
     * id as key.
     *
     * @return Array<string, ProviderInterface>
     */
    public function getApplicableTypeProviders(Mage_Sales_Model_Quote $quote): array {
        return array_filter(
            $this->getTypeProviders($quote->getStore()),
            function($p) use($quote): bool {
                return $p->appliesTo($quote);
            }
        );
    }

    public function getTypeProvider(
        Mage_Core_Model_Store $store,
        string $type
    ): ?ProviderInterface {
        /**
         * @var Array<string, array{model?: ?string}>
         */
        $data = $store->getConfig(self::CONFIG_PROVIDERS) ?: [];

        if(empty($data[$type]["model"])) {
            return null;
        }

        $model = Mage::getSingleton($data[$type]["model"]);

        if( ! $model instanceof ProviderInterface) {
            throw new Exception(sprintf(
                "%s: Model \"%s\" defined in %s/%s/model does not implement %s",
                __METHOD__,
                $data[$type]["model"] ?? "",
                self::CONFIG_PROVIDERS,
                $type,
                ProviderInterface::class
            ));
        }

        return $model;
    }

    public function getAdminhtmlProductEditTab(): array {
        return [
            "label" => $this->__("Points Payment"),
            "title" => $this->__("Points Payment"),
            "url" => Mage::app()->getStore()->getUrl("*/catalog_product_points", ["_current" => true]),
            "class" => "ajax",
        ];
    }
}
