<?php

declare(strict_types=1);

use Points\Core\QuoteAddressTotal;
use Points\Core\ProviderInterface;
use Points\Core\Extension\Quote as QuoteExt;
use Points\Core\Extension\QuoteAddress;

/**
 * Singleton caching the totals calculations.
 */
class Points_Core_Model_Cache_Quote {
    /**
     * @var ?array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}
     */
    private $selected = null;

    /**
     * @var ?Array<array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}>
     */
    private $available = null;

    public function reset(): void {
        $this->available = null;
        $this->selected = null;
    }

    /**
     * @return ?array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}
     */
    public function getSelectedTotal(): ?array {
        if( ! $this->selected) {
            $session = Mage::getSingleton("checkout/session");
            /**
             * @var QuoteExt $quote
             */
            $quote = $session->getQuote();
            $type = $quote->getPointsType();

            if( ! $type) {
                return null;
            }

            $helper = Mage::helper("points_core");
            $provider = $helper->getTypeProvider($quote->getStore(), $type);

            if( ! $provider) {
                return null;
            }

            /**
             * @var QuoteAddress
             */
            $address = $quote->isVirtual() ?
                $quote->getBillingAddress() :
                $quote->getShippingAddress();

            $this->selected = [
                "type" => $type,
                "provider" => $provider,
                "total" => QuoteAddressTotal::fromQuoteAddress($address, $type, $provider),
            ];
        }

        return $this->selected;
    }

    /**
     * @return Array<array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}>
     */
    public function getAvailableTotals(): array {
        if( ! $this->available) {
            $helper = Mage::helper("points_core");
            $session = Mage::getSingleton("checkout/session");
            /**
             * @var QuoteExt $quote
             */
            $quote = $session->getQuote();
            /**
             * @var QuoteAddress
             */
            $address = $quote->isVirtual() ?
                $quote->getBillingAddress() :
                $quote->getShippingAddress();

            $providers = $helper->getApplicableTypeProviders($quote);

            $this->available = array_values(array_map(function(
                string $type,
                ProviderInterface $provider
            ) use($address): array {
                return [
                    "type" => $type,
                    "provider" => $provider,
                    "total" => QuoteAddressTotal::fromQuoteAddress($address, $type, $provider),
                ];
            }, array_keys($providers), $providers));
        }

        return $this->available;
    }
}
