<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use Points\Core\QuoteAddressTotal;
use Points\Core\Points;
use Points\Core\Currency;

class QuoteShipping {
    /**
     * @var Array<QuoteAddressTotal>
     */
    protected $totals;

    /**
     * @var Currency
     */
    protected $currency;

    /**
     * @var ?Currency
     */
    protected $discount;

    /**
     * @param Array<QuoteAddressTotal> $totals
     * @param array{currency:Currency, discount:?Currency} $amounts
     */
    public function __construct(
        array $totals,
        array $amounts
    ) {
        $this->totals = $totals;
        $this->currency = $amounts["currency"];
        $this->discount = $amounts["discount"];
    }

    public function getPoints(): ?QuoteShippingPoints {
        $parts = $this->getPointsParts();

        return $parts ? new QuoteShippingPoints($parts) : null;
    }

    public function getCurrency(): QuoteShippingCurrency {
        return new QuoteShippingCurrency([
            "currency" => $this->currency,
            "discount" => $this->discount,
        ], $this->getPointsParts());
    }

    /**
     * @return ?array{min:Points, max:Points, value:Points, discount:?Points}
     */
    protected function getPointsParts(): ?array {
        $min = array_values(array_filter(array_map(function(QuoteAddressTotal $t): ?Points {
            return $t->getShippingPointsMin();
        }, $this->totals)));
        $max = array_values(array_filter(array_map(function(QuoteAddressTotal $t): ?Points {
            return $t->getShippingPointsMax();
        }, $this->totals)));
        $value = array_values(array_filter(array_map(function(QuoteAddressTotal $t): ?Points {
            return $t->getShippingPointsValue();
        }, $this->totals)));
        $discount = array_values(array_filter(array_map(function(QuoteAddressTotal $t): ?Points {
            return $t->getPointsDiscount()[QuoteAddressTotal::ROW_SHIPPING] ?? null;
        }, $this->totals)));

        // If one is set, the rest should be too, excluding discount
        if(count($min) !== 1 || count($max) !== 1 || count($value) !== 1) {
            return null;
        }

        return [
            "min" => $min[0],
            "max" => $max[0],
            "value" => $value[0],
            "discount" => $discount[0] ?? null,
        ];
    }
}
