<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use MageQL_Core_Model_Price_Abstract;
use MageQL_Core_Model_Context;
use Mage_Tax_Model_Config;

use Points\Core\Currency;
use Points\Core\QuoteAddressTotal;

class QuoteCurrency {
    /**
     * @var Array<QuoteAddressTotal>
     */
    protected $totals;

    /**
     * @param Array<QuoteAddressTotal> $totals
     */
    public function __construct(
        array $totals
    ) {
        $this->totals = $totals;
    }

    public function getMin(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Price_Abstract {
        $incVat = (bool)$ctx->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getTotal = $incVat ? "Points\Core\Amount::totalInclTax" : "Points\Core\Amount::totalExclTax";

        $value = array_sum(array_map(function(QuoteAddressTotal $t): int {
            return array_sum(array_map("Points\Core\Amount::totalInclTax", $t->getPointsValue()));
        }, $this->totals));
        $max = array_sum(array_map(function(QuoteAddressTotal $t): int {
            return array_sum(array_map("Points\Core\Amount::totalInclTax", $t->getPointsMax()));
        }, $this->totals));

        $rate = $value > 0 ? max(1 - $max / $value, 0) : 0;

        return new QuoteCurrencyTotal(
            $this->getExcluded($getTotal) + $this->getIncluded($getTotal) * $rate,
            $incVat,
            $this->getExcludedTax() + $this->getIncludedTax() * $rate
        );
    }

    public function getMax(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Price_Abstract {
        $incVat = (bool)$ctx->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getTotal = $incVat ? "Points\Core\Amount::totalInclTax" : "Points\Core\Amount::totalExclTax";

        $value = array_sum(array_map(function(QuoteAddressTotal $t): int {
            return array_sum(array_map("Points\Core\Amount::totalInclTax", $t->getPointsValue()));
        }, $this->totals));
        $min = array_sum(array_map(function(QuoteAddressTotal $t): int {
            return array_sum(array_map("Points\Core\Amount::totalInclTax", $t->getPointsMin()));
        }, $this->totals));

        $rate = $value > 0 ? (1 - max($min / $value, 0)) : 0;

        return new QuoteCurrencyTotal(
            $this->getExcluded($getTotal) + $this->getIncluded($getTotal) * $rate,
            $incVat,
            $this->getExcludedTax() + $this->getIncludedTax() * $rate
        );
    }

    public function getValue(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Price_Abstract {
        $incVat = (bool)$ctx->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getTotal = $incVat ? "Points\Core\Amount::totalInclTax" : "Points\Core\Amount::totalExclTax";

        return new QuoteCurrencyTotal(
            $this->getExcluded($getTotal) + $this->getIncluded($getTotal),
            $incVat,
            $this->getExcludedTax() + $this->getIncludedTax()
        );
    }

    /**
     * @param callable(Currency):float $getTotal
     */
    protected function getIncluded(callable $getTotal): float {
        return array_sum(array_map(function(QuoteAddressTotal $t) use($getTotal): float {
            return array_sum(array_map($getTotal, array_filter($t->getCurrency(), "Points\Core\Currency::isIncluded")));
        }, $this->totals));
    }

    protected function getIncludedTax(): float {
        return array_sum(array_map(function(QuoteAddressTotal $t): float {
            return array_sum(array_map("Points\Core\Amount::tax", array_filter($t->getCurrency(), "Points\Core\Currency::isIncluded")));
        }, $this->totals));
    }

    /**
     * @param callable(Currency):float $getTotal
     */
    protected function getExcluded(callable $getTotal): float {
        return array_sum(array_map(function(QuoteAddressTotal $t) use($getTotal): float {
            return array_sum(array_map($getTotal, array_filter($t->getCurrency(), "Points\Core\Currency::isNotIncluded")));
        }, $this->totals));
    }

    protected function getExcludedTax(): float {
        return array_sum(array_map(function(QuoteAddressTotal $t): float {
            return array_sum(array_map("Points\Core\Amount::tax", array_filter($t->getCurrency(), "Points\Core\Currency::isNotIncluded")));
        }, $this->totals));
    }
}
