<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use MageQL_Core_Model_Context;
use Mage_Sales_Model_Quote;

use Points\Core\Currency;
use Points\Core\Extension\Quote as QuoteExt;
use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;

class Quote {
    /**
     * @var string
     */
    protected $type;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @var QuoteExt
     */
    protected $quote;

    /**
     * @var ?Array<QuoteAddressTotal>
     */
    protected $totals;

    /**
     * @param QuoteExt $quote
     */
    public function __construct(
        string $type,
        ProviderInterface $provider,
        Mage_Sales_Model_Quote $quote
    ) {
        $this->type = $type;
        $this->provider = $provider;
        $this->quote = $quote;
    }

    /**
     * @return Array<QuoteAddressTotal>
     */
    protected function getTotals() {
        if($this->totals === null) {
            $this->totals = [];

            foreach($this->quote->getAddressesCollection() as $addr) {
                $this->totals[] = QuoteAddressTotal::fromQuoteAddress($addr, $this->type, $this->provider);
            }
        }

        return $this->totals;
    }

    public function getId(): string {
        return $this->type;
    }

    public function getLabel(array $args, MageQL_Core_Model_Context $ctx): string {
        return $this->provider->getLabel($ctx->getStore());
    }

    public function getCurrencyRequired(): bool {
        $addressTotals = $this->getTotals();

        $excluded = array_sum(array_map(function(QuoteAddressTotal $t): float {
            return array_sum(array_map("Points\Core\Amount::totalInclTax", array_filter($t->getCurrency(), "Points\Core\Currency::isNotIncluded")));
        }, $addressTotals));

        if($excluded > 0) {
            return true;
        }

        $value = array_sum(array_map(function(QuoteAddressTotal $t): int {
            return array_sum(array_map("Points\Core\Amount::totalInclTax", $t->getPointsValue()));
        }, $addressTotals));
        $max = array_sum(array_map(function(QuoteAddressTotal $t): int {
            return array_sum(array_map("Points\Core\Amount::totalInclTax", $t->getPointsMax()));
        }, $addressTotals));

        return $max < $value;
    }

    public function getPoints(): QuotePoints {
        return new QuotePoints($this->getTotals(), $this->quote, $this->provider);
    }

    public function getCurrency(): QuoteCurrency {
        return new QuoteCurrency($this->getTotals());
    }

    public function getShipping(): ?QuoteShipping {
        $amounts = $this->getShippingAmounts();

        if( ! $amounts) {
            return null;
        }

        return new QuoteShipping($this->getTotals(), $amounts);
    }

    public function getDiscount(): ?QuoteDiscount {
        // TODO: This should be easier if we just use a more generic total model
        $totals = $this->getTotals();
        $discount = array_values(array_filter(array_merge(...array_map(function(QuoteAddressTotal $t): array {
            return $t->getDiscount();
        }, $totals))));
        $discountPoints = array_values(array_filter(array_merge(...array_map(function(QuoteAddressTotal $t): array {
            return $t->getPointsDiscount();
        }, $totals))));

        if(empty($discount)) {
            return null;
        }

        return new QuoteDiscount($discount, $discountPoints);
    }

    /**
     * @return ?array{currency:Currency, discount:?Currency}
     */
    protected function getShippingAmounts(): ?array {
        $totals = $this->getTotals();

        $currency = array_values(array_filter(array_map(function(QuoteAddressTotal $t): ?Currency {
            return $t->getShippingCurrency();
        }, $totals)));
        $discount = array_values(array_filter(array_map(function(QuoteAddressTotal $t): ?Currency {
            return $t->getDiscount()[QuoteAddressTotal::ROW_SHIPPING] ?? null;
        }, $totals)));

        // 1 or 0 shipping values
        if(count($currency) !== 1) {
            return null;
        }

        return [
            "currency" => $currency[0],
            "discount" => $discount[0] ?? null,
        ];
    }
}
