<?php

declare(strict_types=1);

/**
 * @method int getEntityId()
 * @method int getWebsiteId()
 * @method void setWebsiteId(int $id)
 * @method int getCustomerGroupId()
 * @method void setCustomerGroupId(int $id)
 * @method string getType()
 * @method void setType(string $type)
 * @method int getProductId()
 * @method void setProductId(int $productId)
 * @method int getPrice()
 * @method void setPrice(int $price)
 * @method int getMinPrice()
 * @method void setMinPrice(int $price)
 * @method int getMaxPrice()
 * @method void setMaxPrice(int $price)
 * @method string getCreatedAt()
 * @method string getUpdatedAt()
 */
class Points_Core_Model_Product_Index extends Mage_Index_Model_Indexer_Abstract {
    /**
     * Data key for matching result to be saved in
     */
    const EVENT_MATCH_RESULT_KEY = "points_core_product_price_match_result";

    const KEY_REINDEX_PRODUCT_IDS = "points_core_product_price_reindex_parent_ids";
    const KEY_REINDEX_ALL = "points_core_product_price_reindex_all";

    const ATTRIBUTES = [
        "price",
        "min_price",
        "max_price",
    ];

    const PRODUCT_ATTRIBUTES = [
        "status",
        "required_options",
        "tax_class_id",
        "force_reindex_required",
    ];

    /**
     * Matched Entities instruction array
     *
     * @var array
     */
    protected $_matchedEntities = [
        Mage_Catalog_Model_Product::ENTITY => [
            Mage_Index_Model_Event::TYPE_DELETE,
            Mage_Index_Model_Event::TYPE_SAVE,
            Mage_Index_Model_Event::TYPE_MASS_ACTION,
        ],
        Mage_Core_Model_Config_Data::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
        ],
        Mage_Customer_Model_Group::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
        ],
        Points_Core_Model_Product_Price::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
            Mage_Index_Model_Event::TYPE_DELETE,
        ],
    ];

    const RELATED_CONFIG_SETTINGS = [
        Mage_Catalog_Helper_Data::XML_PATH_PRICE_SCOPE,
        Mage_CatalogInventory_Model_Stock_Item::XML_PATH_MANAGE_STOCK
    ];

    public function _construct(): void {
        parent::_construct();

        $this->_init("points_core/product_index");
    }

    public function getName() {
        return Mage::helper("points_core")->__("Product Point Prices");
    }

    public function getDescription() {
        return Mage::helper("points_core")->__("Index product point prices");
    }

    public function matchEvent(Mage_Index_Model_Event $event) {
        $data = $event->getNewData();

        if(isset($data[self::EVENT_MATCH_RESULT_KEY])) {
            return $data[self::EVENT_MATCH_RESULT_KEY];
        }

        if ($event->getEntity() == Mage_Core_Model_Config_Data::ENTITY) {
            /**
             * @var ?Varien_Object
             */
            $data = $event->getDataObject();

            if($data && in_array($data->getPath(), self::RELATED_CONFIG_SETTINGS)) {
                $result = $data->isValueChanged();
            }
            else {
                $result = false;
            }
        }
        elseif($event->getEntity() == Mage_Customer_Model_Group::ENTITY) {
            $result = $event->getDataObject() && $event->getDataObject()->isObjectNew();
        }
        else {
            $result = parent::matchEvent($event);
        }

        $event->addNewData(self::EVENT_MATCH_RESULT_KEY, $result);

        return $result;
    }

    /**
     * @return void
     */
    protected function _registerEvent(Mage_Index_Model_Event $event) {
        $event->addNewData(self::EVENT_MATCH_RESULT_KEY, true);

        $entity = $event->getEntity();

        switch($entity) {
        case Mage_Core_Model_Config_Data::ENTITY:
        case Mage_Customer_Model_Group::ENTITY:
            $process = $event->getProcess();

            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);

            break;

        case Points_Core_Model_Product_Price::ENTITY:
            /** @var Points_Core_Model_Product_Price */
            $price = $event->getDataObject();
            $reindexPrice = false;

            foreach(self::ATTRIBUTES as $attributeCode) {
                $reindexPrice = $reindexPrice || $price->dataHasChangedFor($attributeCode);
            }

            if($reindexPrice) {
                $event->addNewData(self::KEY_REINDEX_PRODUCT_IDS, [$price->getProductId()]);
            }

            break;

        case Mage_Catalog_Model_Product::ENTITY:
            switch($event->getType()) {
                case Mage_Index_Model_Event::TYPE_DELETE:
                    // Reindex parent product
                    /**
                     * @var Mage_Catalog_Model_Product $product
                     */
                    $product = $event->getDataObject();
                    /**
                     * @var Points_Core_Model_Resource_Product_Index $resource
                     */
                    $resource = $this->_getResource();

                    $parentIds = $resource->getProductParentsByChild($product->getId());

                    if($parentIds) {
                        // Maybe just use products
                        $event->addNewData(self::KEY_REINDEX_PRODUCT_IDS, $parentIds);
                    }

                    break;

                case Mage_Index_Model_Event::TYPE_SAVE:
                    /* @var Mage_Catalog_Model_Product $product */
                    $product = $event->getDataObject();

                    $reindexPrice = $product->getIsRelationsChanged()
                        || $product->getIsCustomOptionChanged()
                        || $product->dataHasChangedFor("tier_price_changed")
                        || $product->getIsChangedWebsites()
                        || $product->getForceReindexRequired();

                    foreach(self::PRODUCT_ATTRIBUTES as $attributeCode) {
                        $reindexPrice = $reindexPrice || $product->dataHasChangedFor($attributeCode);
                    }

                    if($reindexPrice) {
                        $event->addNewData(self::KEY_REINDEX_PRODUCT_IDS, [$product->getId()]);
                    }

                    break;

                case Mage_Index_Model_Event::TYPE_MASS_ACTION:
                    /* @var Varien_Object $actionObject */
                    $actionObject = $event->getDataObject();
                    $reindexPrice = false;

                    // check if attributes changed
                    $attrData = $actionObject->getAttributesData();

                    if(is_array($attrData)) {
                        foreach(self::PRODUCT_ATTRIBUTES as $attributeCode) {
                            if(array_key_exists($attributeCode, $attrData)) {
                                $reindexPrice = true;
                                break;
                            }
                        }
                    }

                    // check changed websites
                    if($actionObject->getWebsiteIds()) {
                        $reindexPrice = true;
                    }

                    if($reindexPrice) {
                        $event->addNewData(self::KEY_REINDEX_PRODUCT_IDS, $actionObject->getProductIds());
                    }

                    break;
            }

            break;
        }
    }

    /**
     * @return mixed
     */
    protected function _processEvent(Mage_Index_Model_Event $event) {
        $data = $event->getNewData();

        if( ! empty($data[self::KEY_REINDEX_ALL])) {
            return $this->reindexAll();
        }

        if( ! empty($data[self::KEY_REINDEX_PRODUCT_IDS])) {
            /**
             * @var Points_Core_Model_Resource_Product_Index $resource
             */
            $resource = $this->_getResource();

            return $resource->reindexProductIds($data[self::KEY_REINDEX_PRODUCT_IDS]);
        }
    }

    protected function _beforeSave() {
        parent::_beforeSave();

        $this->setData("updated_at", date("Y-m-d\\TG:i:s"));

        return $this;
    }
}
