<?php

declare(strict_types=1);

namespace Points\Core;

use function Fixtures\loadPoints;

use Mage;
use Mage_Customer_Model_Customer;
use Mage_Customer_Model_Group;
use Points_Core_Helper_Data;
use Throwable;
use Varien_Object;
use Varien_Profiler;

use Points\Core\ProviderInterface;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Fixtures\Customer;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/../Fixtures/Points.php";

class TestTest extends TestCase {
    use MatchesSnapshots;

    public static function setUpBeforeClass(): void {
        loadPoints();
    }

    public function setUp(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    public function testProductDetail(): void {
        MagentoManager::init();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            productBySku(sku: "test-simple") {
                sku
                pointsPrices {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    price { incVat exVat vat }
                    maximumCurrency { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductDetail2(): void {
        MagentoManager::init();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            productBySku(sku: "test-virtual") {
                sku
                pointsPrices {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    price { incVat exVat vat }
                    maximumCurrency { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductDetail3(): void {
        MagentoManager::init();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            productBySku(sku: "test-config") {
                sku
                pointsPrices {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    price { incVat exVat vat }
                    maximumCurrency { incVat exVat vat }
                }
                ...on ProductDetailConfigurable {
                    configOptions {
                        items {
                            product {
                                sku
                            }
                            pointsPrices {
                                id
                                label
                                minimum { incVat exVat vat }
                                maximum { incVat exVat vat }
                                price { incVat exVat vat }
                                maximumCurrency { incVat exVat vat }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductList(): void {
        MagentoManager::init();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            products {
                items {
                    sku
                    pointsPrices {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        price { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                    ...on ListProductConfigurable {
                        configOptions {
                            items {
                                product {
                                    sku
                                }
                                pointsPrices {
                                    id
                                    label
                                    minimum { incVat exVat vat }
                                    maximum { incVat exVat vat }
                                    price { incVat exVat vat }
                                    maximumCurrency { incVat exVat vat }
                                }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductDetailLoggedIn(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer");
        $session = Mage::getSingleton("customer/session");

        $customer->setWebsiteId($store->getWebsiteId());
        $customer->loadByEmail("test-customer@example.com");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            productBySku(sku: "test-simple") {
                sku
                pointsPrices {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    price { incVat exVat vat }
                    maximumCurrency { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductDetailLoggedIn2(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer");
        $session = Mage::getSingleton("customer/session");

        $customer->setWebsiteId($store->getWebsiteId());
        $customer->loadByEmail("test-customer@example.com");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            productBySku(sku: "test-virtual") {
                sku
                pointsPrices {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    price { incVat exVat vat }
                    maximumCurrency { incVat exVat vat }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductDetailLoggedIn3(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer");
        $session = Mage::getSingleton("customer/session");

        $customer->setWebsiteId($store->getWebsiteId());
        $customer->loadByEmail("test-customer@example.com");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            productBySku(sku: "test-config") {
                sku
                pointsPrices {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    price { incVat exVat vat }
                    maximumCurrency { incVat exVat vat }
                }
                ...on ProductDetailConfigurable {
                    configOptions {
                        items {
                            product {
                                sku
                            }
                            pointsPrices {
                                id
                                label
                                minimum { incVat exVat vat }
                                maximum { incVat exVat vat }
                                price { incVat exVat vat }
                                maximumCurrency { incVat exVat vat }
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testProductListLoggedIn(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer");
        $session = Mage::getSingleton("customer/session");

        $customer->setWebsiteId($store->getWebsiteId());
        $customer->loadByEmail("test-customer@example.com");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            products {
                items {
                    sku
                    pointsPrices {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        price { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testNone(): void {
        MagentoManager::init();

        $quote = Mage::getModel("sales/quote");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 2,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $billing = $quote->getBillingAddress();
        $shipping = $quote->getShippingAddress();

        $this->assertEquals(29.53, $shipping->getGrandTotal());
        $this->assertEquals(25.86, $shipping->getSubtotal());
        $this->assertEquals(10.69, $shipping->getShippingAmount());
        $this->assertEquals(2.68, $shipping->getShippingTaxAmount());
        $this->assertEquals(5.91, $shipping->getTaxAmount());
        $this->assertEquals(-16.16, $shipping->getDiscountAmount());
        $this->assertEquals(0, $billing->getGrandTotal());
        $this->assertEquals(0, $billing->getSubtotal());
        $this->assertEquals(0, $billing->getShippingAmount());
        $this->assertEquals(0, $billing->getShippingTaxAmount());
        $this->assertEquals(0, $billing->getTaxAmount());
        $this->assertEquals(0, $billing->getDiscountAmount());
    }

    public function testZero(): void {
        MagentoManager::init();

        $quote = Mage::getModel("sales/quote");
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 2,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setData(Points_Core_Helper_Data::SALES_POINTS_TYPE_KEY, "TEST");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $billing = $quote->getBillingAddress();
        $shipping = $quote->getShippingAddress();

        $this->assertEquals(29.53, $shipping->getGrandTotal());
        $this->assertEquals(25.86, $shipping->getSubtotal());
        $this->assertEquals(10.69, $shipping->getShippingAmount());
        $this->assertEquals(2.68, $shipping->getShippingTaxAmount());
        $this->assertEquals(5.91, $shipping->getTaxAmount());
        $this->assertEquals(-16.16, $shipping->getDiscountAmount());
        $this->assertEquals(0, $billing->getGrandTotal());
        $this->assertEquals(0, $billing->getSubtotal());
        $this->assertEquals(0, $billing->getShippingAmount());
        $this->assertEquals(0, $billing->getShippingTaxAmount());
        $this->assertEquals(0, $billing->getTaxAmount());
        $this->assertEquals(0, $billing->getDiscountAmount());
    }

    public function testAvailablePointPurchaseMethods(): void {
        MagentoManager::init();

        $quote = Mage::getModel("sales/quote");
        $mockProvider = $this->createMock(ProviderInterface::class);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->with($quote->getShippingAddress())
            ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 2,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $checkoutSession = Mage::getSingleton("checkout/session");

        // reload quote
        $checkoutSession->setQuoteId($quote->getId());

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::unregister("_singleton/custom/provider");
        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
            quote {
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    /*
    public function testVerify2(): void {
        MagentoManager::init();

        $quote = Mage::getModel("sales/quote");

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $billing = $quote->getBillingAddress();
        $shipping = $quote->getShippingAddress();

        $this->assertEquals($shipping->getSubtotal(), 9.87);
        $this->assertEquals($shipping->getShippingAmount(), 10.7);
        $this->assertEquals($shipping->getShippingTaxAmount(), 2.67);
        $this->assertEquals($shipping->getTaxAmount(), 2.47 + 2.67);
        $this->assertEquals($shipping->getGrandTotal(), 9.87 + 10.7 + 2.47 + 2.67);
        $this->assertEquals($shipping->getDiscountAmount(), 0);
        $this->assertEquals($billing->getGrandTotal(), 0);
        $this->assertEquals($billing->getSubtotal(), 0);
        $this->assertEquals($billing->getShippingAmount(), 0);
        $this->assertEquals($billing->getShippingTaxAmount(), 0);
        $this->assertEquals($billing->getTaxAmount(), 0);
        $this->assertEquals($billing->getDiscountAmount(), 0);
    }

    public function testVerify2(): void {
        MagentoManager::init();

        $quote = Mage::getModel("sales/quote");

        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $billing = $quote->getBillingAddress();
        $shipping = $quote->getShippingAddress();

        $this->assertEquals($shipping->getSubtotal(), 9.87);
        $this->assertEquals($shipping->getShippingAmount(), 10.7);
        $this->assertEquals($shipping->getShippingTaxAmount(), 2.67);
        $this->assertEquals($shipping->getTaxAmount(), 2.47 + 2.67);
        $this->assertEquals($shipping->getGrandTotal(), 9.87 + 10.7 + 2.47 + 2.67);
        $this->assertEquals($shipping->getDiscountAmount(), 0);
        $this->assertEquals($billing->getGrandTotal(), 0);
        $this->assertEquals($billing->getSubtotal(), 0);
        $this->assertEquals($billing->getShippingAmount(), 0);
        $this->assertEquals($billing->getShippingTaxAmount(), 0);
        $this->assertEquals($billing->getTaxAmount(), 0);
        $this->assertEquals($billing->getDiscountAmount(), 0);
    }

    public function _testVerify2(): void {
        MagentoManager::init();

        $quote = Mage::getModel("sales/quote");

        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        // We have to use load, cannot use loadByAttribute or similar
        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        // Plain quantity request
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setCouponCode("PHPUnitTest1");

        $address = $quote->getShippingAddress();

        $address->setShippingMethod("flatrate_flatrate");
        $address->setCollectShippingRates(true);

        // Got to set this manually
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        var_dump($quote->getCouponCode());
        var_dump($quote->getGrandTotal());

        // TODO: Assertions
    }
     */

    public function testFilterableBy(): void {
        MagentoManager::init();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products {
                    filterableBy {
                        __typename
                        code
                        label
                        ... on ProductFilterBucket { values { value } }
                        ... on ProductFilterPrice { incVat { min max } exVat { min max } }
                        ... on ProductFilterRange { range {  min max } }
                    }
                    items {
                        sku
                        pointsPrices {
                            id
                            label
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            price { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                }
            }'
        ));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testFilterMin(): void {
        MagentoManager::init();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "points:TEST", minValue: 1000 } ]
                ) {
                    filterableBy {
                        __typename
                        code
                        label
                        ... on ProductFilterBucket { values { value } }
                        ... on ProductFilterPrice { incVat { min max } exVat { min max } }
                        ... on ProductFilterRange { range {  min max } }
                    }
                    items {
                        sku
                        pointsPrices {
                            id
                            label
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            price { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                }
            }'
        ));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }

    public function testFilterMax(): void {
        MagentoManager::init();

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("isEnabled")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $res = MagentoManager::runRequest(new Request("POST /graphql", ["Content-Type" => "application/graphql"], 'query {
                products (
                    filter: [ { code: "points:TEST", maxValue: 1000 } ]
                ) {
                    filterableBy {
                        __typename
                        code
                        label
                        ... on ProductFilterBucket { values { value } }
                        ... on ProductFilterPrice { incVat { min max } exVat { min max } }
                        ... on ProductFilterRange { range {  min max } }
                    }
                    items {
                        sku
                        pointsPrices {
                            id
                            label
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            price { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                }
            }'
        ));

        $this->assertMatchesJsonSnapshot($res->getBody());
        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
    }
}
