<?php

declare(strict_types=1);

use Points\Core\Currency;
use Points\Core\Points;
use Points\Core\ProviderInterface;
use Points\Core\Total;

class Points_Core_Model_Schema_QuoteItemPointPayment {
    /**
     * @var string
     */
    protected $type;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @var Total
     */
    protected $total;

    /**
     * @var Mage_Sales_Model_Quote_Item $item
     */
    protected $item;

    public function __construct(
        string $type,
        ProviderInterface $provider,
        Total $total,
        Mage_Sales_Model_Quote_Item $item
    ) {
        $this->type = $type;
        $this->provider = $provider;
        $this->total = $total;
        $this->item = $item;
    }

    public function getId(): string {
        return $this->type;
    }

    public function getLabel(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): string {
        return $this->provider->getLabel($ctx->getStore());
    }

    public function getPoints(): Points_Core_Model_Schema_QuoteTotal {
        return new Points_Core_Model_Schema_QuoteTotal(
            [$this->total],
            function(Total $t): array {
                return $t->getPointsValue();
            }
        );
    }

    public function getMinimum(): Points_Core_Model_Schema_QuoteTotal {
        return new Points_Core_Model_Schema_QuoteTotal(
            [$this->total],
            function(Total $t): array {
                return $t->getPointsMin();
            }
        );
    }

    public function getMaximum(): Points_Core_Model_Schema_QuoteTotal {
        return new Points_Core_Model_Schema_QuoteTotal(
            [$this->total],
            function(Total $t): array {
                return $t->getPointsMax();
            }
        );
    }

    public function getMaximumCurrency(array $unusedArgs, MageQL_Core_Model_Context $ctx): Varien_Object {
        $value = array_sum(array_map("Points\Core\Amount::totalInclTax", $this->total->getPointsValue()));
        $minimum = array_sum(array_map("Points\Core\Amount::totalInclTax", $this->total->getPointsMin()));

        $percent = ($value - $minimum) / $value;
        $store = $ctx->getStore();

        $incVat = (float)$store->roundPrice($this->item->getRowTotal()
            + $this->item->getTaxAmount()
            + $this->item->getHiddenTaxAmount()
            - $this->item->getDiscountAmount());

        return new Varien_Object([
            "ex_vat" => (float)$store->roundPrice($percent * ($this->item->getRowTotal() - $this->item->getDiscountAmount())),
            "inc_vat" => (float)$store->roundPrice($percent * $incVat),
            "vat" => (float)$store->roundPrice($percent * $this->item->getTaxAmount()),
        ]);
    }
}
