<?php

declare(strict_types=1);

use function Points\Core\calculateRoundingSpread;

use Points\Core\QuoteAddressTotal;
use Points\Core\Extension\QuoteAddress;
use Points\Core\Extension\QuoteAddressItem;
use Points\Core\Extension\Quote;

/**
 * @psalm-type Limit array{min:int, max:int}
 */
class Points_Core_Model_Sales_Total_Quote_Points extends Mage_Sales_Model_Quote_Address_Total_Abstract {
    protected $_code = "points";

    public function collect(Mage_Sales_Model_Quote_Address $address) {
        parent::collect($address);

        /**
         * @var QuoteAddress $address
         */
        $address->setPointsPoints(0);
        $address->setPointsTaxPoints(0);
        $address->setBasePointsAmount(0);
        $address->setBasePointsTaxAmount(0);
        $address->setPointsAmount(0);
        $address->setPointsTaxAmount(0);
        $address->unsPointsType();

        $address->setPointsPointsTotal(0);
        $address->setPointsTaxPointsTotal(0);
        $address->setPointsAmountIncluded(0);
        $address->setPointsTaxAmountIncluded(0);
        $address->setPointsAmountExcluded(0);
        $address->setPointsTaxAmountExcluded(0);
        $address->setPointsShippingPointsValue(null);
        $address->setPointsShippingTaxPointsValue(null);

        $items = $this->_getAddressItems($address);

        if(count($items) === 0) {
            // It is already reset in observer
            return $this;
        }

        /**
         * @var QuoteAddressItem $i
         */
        foreach($items as $i) {
            // Reset to make sure we do not populate if it does not have a point price
            $i->setPointsRowPointsValue(null);
            $i->setPointsRowTaxPointsValue(null);
        }

        $helper = Mage::helper("points_core");
        /**
         * @var Quote $quote
         */
        $quote = $address->getQuote();
        $type = $quote->getPointsType();

        if( ! $type) {
            return $this;
        }

        $store = $quote->getStore();
        $customer = $quote->getCustomer();
        $provider = $helper->getTypeProvider($store, $type);

        if( ! $provider ||
            ! $provider->appliesTo($quote) ||
            ! $customer->getId() ||
            ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            return $this;
        }

        $totals = QuoteAddressTotal::fromQuoteAddress($address, $type, $provider);
        $incVat = $quote->getPointsWantedIncludesTax() ?? (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getAmountTotal = $incVat ? "Points\Core\Amount::totalInclTax" : "Points\Core\Amount::totalExclTax";

        $customerPoints = $provider->getCustomerPointsBalance($store, $customer);
        $customerPointsIncludesTax = $provider->getCustomerPointsBalanceIncludesTax($store, $customer);

        /**
         * @var QuoteAddressItem $i
         */
        foreach($items as $i) {
            $value = $totals->getRowPointsValue($i);

            $i->setPointsRowPointsValue($value ? $value->getTotalExclTax() : null);
            $i->setPointsRowTaxPointsValue($value ? $value->getTax() : null);
        }

        $pointsTotal = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsMax = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMax()));
        $pointsTax = array_sum(array_map("Points\Core\Amount::tax", $totals->getPointsValue()));

        if($pointsTotal - $pointsTax < 1) {
            // No points, nothing to do
            return $this;
        }

        $totalLimit = Mage::getResourceModel("points_core/limit_total")->getMatchingLimit($store, $customer->getGroupId(), $type);
        $orderLimit = Mage::getResourceModel("points_core/limit_order")->getMatchingLimit($store, $customer->getGroupId(), $type);

        // Rate from a total without tax
        $pointTaxRate = $incVat ? $pointsTax / ($pointsTotal - $pointsTax) : $pointsTax / $pointsTotal;
        // Tax multiplier from a total including tax
        $taxMultiplier = $pointTaxRate / (1 + $pointTaxRate);

        $availablePoints = min(
            // Reduce the available points by the tax rate if they include, otherwise increase them
            $customerPointsIncludesTax ? (
                $incVat ? $customerPoints : (int)floor($customerPoints * (1 - $taxMultiplier))
            ) : (
                $incVat ? (int)floor($customerPoints * (1 + $pointTaxRate)) : $customerPoints
            ),
            $pointsMax,
            $totalLimit ? (int)$totalLimit->getLimit() - $totalLimit->getSpentDuringLastWindow($customer) : $pointsTotal,
            $orderLimit ? $orderLimit->getTotalMax($pointsTotal) : $pointsTotal
        );

        $minPoints = max(
            0,
            $orderLimit ? $orderLimit->getMinValue() ?: 0 : 0
        );

        // Includes or excludes tax depending on incVat
        $requestedPoints = min($availablePoints, (int)$quote->getPointsWanted());

        if($requestedPoints < $minPoints) {
            // Exit since we can't proceed
            return $this;
        }

        $included = array_filter($totals->getCurrency(), "Points\Core\Currency::isIncluded");
        $includedAmount = array_sum(array_map($getAmountTotal, $included));
        $includedTax = array_sum(array_map("Points\Core\Amount::tax", $included));

        $excluded = array_filter($totals->getCurrency(), "Points\Core\Currency::isNotIncluded");
        $excludedAmount = array_sum(array_map($getAmountTotal, $excluded));
        $excludedTax = array_sum(array_map("Points\Core\Amount::tax", $excluded));

        if($includedAmount + $excludedAmount < 0.01) {
            // TODO: Make sure to skip this if the products can only be bought using points
            Mage::log(sprintf(
                "%s: Zero included + excluded currency amount for quote '%d' and address '%d', items: %s",
                __METHOD__,
                (int)$quote->getId(),
                (int)$address->getId(),
                json_encode(array_map(function(Mage_Sales_Model_Quote_Address_Item $i): array {
                    /**
                     * For some reason address_item does not specify these
                     * fields but inherits them anyway:
                     *
                     * @var Mage_Sales_Model_Quote_Item $i
                     */
                    return [
                        "id" => $i->getId(),
                        "sku" => $i->getSku(),
                        "qty" => $i->getQty(),
                        "productId" => $i->getProductId(),
                        "productType" => $i->getProductType(),
                    ];
                }, $items))
            ));

            return $this;
        }

        // Amount of the quote which is covered by points
        $fracIncluded = $includedAmount / ($includedAmount + $excludedAmount);
        // Amount of the total points value to spend
        $fracPoints = $requestedPoints / $pointsTotal;

        // addressBaseTotal includes tax
        $addressBaseTotal = array_sum($address->getAllBaseTotalAmounts());
        $baseTotalAmount = $store->roundPrice(($addressBaseTotal * $fracIncluded) * $fracPoints);
        $totalAmount = $store->convertPrice($baseTotalAmount, false);

        $baseAmount = $this->clampTax($this->spreadPrice([
            "value" => $baseTotalAmount * (1 - $taxMultiplier),
            "tax" => $baseTotalAmount * $taxMultiplier,
        ]), $address->getBaseTotalAmount("tax"));
        $amount = $this->clampTax($this->spreadPrice([
            "value" => $totalAmount * (1 - $taxMultiplier),
            "tax" => $totalAmount * $taxMultiplier,
        ]), $address->getTotalAmount("tax"));

        if($incVat) {
            $pointParts = calculateRoundingSpread([
                "value" => $requestedPoints * (1 - $taxMultiplier),
                "tax" => $requestedPoints * $taxMultiplier,
            ]);

            $address->setPointsPoints($pointParts["value"]);
            $address->setPointsTaxPoints($pointParts["tax"]);
        }
        else {
            $address->setPointsPoints($requestedPoints);
            $address->setPointsTaxPoints((int)ceil($requestedPoints * $pointTaxRate));
        }

        $address->setTotalAmount($this->getCode(), -$amount["value"]);
        $address->setBaseTotalAmount($this->getCode(), -$baseAmount["value"]);
        // Reduce tax with what we paid
        $address->addTotalAmount("tax", -$amount["tax"]);
        $address->addBaseTotalAmount("tax", -$baseAmount["tax"]);
        $address->setBasePointsTaxAmount($baseAmount["tax"]);
        $address->setPointsTaxAmount($amount["tax"]);
        $address->setPointsType($type);

        $address->setPointsPointsTotal($incVat ? $pointsTotal - $pointsTax : $pointsTotal);
        $address->setPointsTaxPointsTotal($pointsTax);
        $address->setPointsAmountIncluded($incVat ? $includedAmount - $includedTax : $includedAmount);
        $address->setPointsTaxAmountIncluded($includedTax);
        $address->setPointsAmountExcluded($incVat ? $excludedAmount - $excludedTax : $excludedAmount);
        $address->setPointsTaxAmountExcluded($excludedTax);

        $shipping = $totals->getShippingPointsValue();

        if($shipping) {
            $address->setPointsShippingPointsValue($shipping->getTotalExclTax());
            $address->setPointsShippingTaxPointsValue($shipping->getTax());
        }

        $address->setSelectedPointsTotalInstance($totals);

        $quote->setPointsWanted((int)$quote->getPointsWanted() - $requestedPoints);

        return $this;
    }

    public function fetch(Mage_Sales_Model_Quote_Address $address) {
        /**
         * @var QuoteAddress $address
         */
        $address->addTotal([
            "code"           => $this->getCode(),
            "value"          => $address->getPointsAmount(),
            "value_incl_tax" => $address->getPointsAmount() + $address->getPointsTaxAmount(),
            "value_excl_tax" => $address->getPointsAmount(),
        ]);

        return [];
    }

    /**
     * @var Array<string, float> $prices
     * @return Array<string, float>
     */
    public function spreadPrice(array $prices): array {
        return array_map(function(int $a): float {
            return round($a / 100, 2);
        }, calculateRoundingSpread(array_map(
            /**
             * @param float $a
             */
            function($a): float {
                return $a * 100;
            },
            $prices
        )));
    }

    /**
     * Ensures the tax value does not exceed the maximum tax on the address.
     *
     * @param float $maxTax
     */
    public function clampTax(array $data, $maxTax): array {
        $diff = $data["tax"] - $maxTax;

        if($diff > 0) {
            $data["tax"] -= $diff;
            $data["value"] += $diff;
        }

        return $data;
    }
}
