<?php

declare(strict_types=1);

use function Points\Core\getTime;

/**
 * A limit for the total amount of points spent in a time window.
 *
 * @method int getStoreId()
 * @method void setStoreId(int $storeId)
 * @method int getCustomerGroupId()
 * @method void setCustomerGroupId(int $customerGroupId)
 * @method string getType()
 * @method void setType(string $type)
 *
 * @method string getTimeInterval()
 * @method void setTimeInterval(string $interval)
 * @method ?int getLimit()
 * @method void setLimit(?int $price)
 *
 * @method string getCreatedAt()
 * @method string getUpdatedAt()
 */
class Points_Core_Model_Limit_Total extends Mage_Core_Model_Abstract {
    const INTERVAL_WEEK = "WEEK";
    const INTERVAL_MONTH = "MONTH";
    const INTERVAL_QUARTER = "QUARTER";
    const INTERVAL_YEAR = "YEAR";

    public function _construct(): void {
        parent::_construct();

        $this->_init("points_core/limit_total");
    }

    /**
     * @param int $customerGroupId
     * @return $this
     */
    public function loadByStoreTypeCustomerGroupId(
        Mage_Core_Model_Store $store,
        string $type,
        $customerGroupId = Mage_Customer_Model_Group::NOT_LOGGED_IN_ID
    ) {
        /**
         * @var Points_Core_Model_Resource_Limit_Total
         */
        $resource = $this->_getResource();

        $resource->loadByStoreTypeCustomerGroupId($this, $store, $type, $customerGroupId);

        return $this;
    }

    public function setStore(Mage_Core_Model_Store $store): void {
        $this->setStoreId($store->getId());
    }

    public function setCustomerGroup(Mage_Customer_Model_Group $group): void {
        $this->setCustomerGroupId($group->getId());
    }

    public function getPrevResetDatetime(): string {
        // we use a namespaced time here
        $now = getTime();

        switch($this->getTimeInterval()) {
        case self::INTERVAL_WEEK:
            return date("Y-m-d", strtotime("Monday this week", $now));
        case self::INTERVAL_MONTH:
            return date("Y-m-01", $now);
        case self::INTERVAL_QUARTER:
            $offset = (3 - date("n", $now) % 3) % 3;

            return date("Y-m-01", strtotime("first day of +$offset month midnight", $now));
        case self::INTERVAL_YEAR:
            return date("Y-m-01", $now);
        default:
            throw new Exception(sprintf(
                "%s: Invalid time interval '%s'",
                __METHOD__,
                $this->getTimeInterval()
            ));
        }
    }

    public function getNextResetDatetime(): string {
        // we use a namespaced time here
        $now = getTime();

        switch($this->getTimeInterval()) {
        case self::INTERVAL_WEEK:
            return date("Y-m-d", strtotime("Monday next week", $now));
        case self::INTERVAL_MONTH:
            return date("Y-m-01", strtotime("+1 month", $now));
        case self::INTERVAL_QUARTER:
            $offset = (3 - date("n", $now) % 3) % 3 + 1;

            return date("Y-m-01", strtotime("first day of +$offset month midnight", $now));
        case self::INTERVAL_YEAR:
            return date("Y-m-01", strtotime("+1 year", $now));
        default:
            throw new Exception(sprintf(
                "%s: Invalid time interval '%s'",
                __METHOD__,
                $this->getTimeInterval()
            ));
        }
    }

    public function getSpentDuringLastWindow(Mage_Customer_Model_Customer $customer): int {
        $resource = Mage::getResourceSingleton("sales/order");
        $conn = $resource->getReadConnection();

        $query = $conn->select()
            ->from(["o" => $resource->getTable("sales/order")], [])
            ->where("state NOT IN (?)", ["canceled", "closed"])
            ->where("customer_id = ?", $customer->getId())
            ->where("points_type = ?", $this->getType())
            ->where("created_at >= ?", $this->getPrevResetDatetime())
            ->columns(["sum" => new Zend_Db_Expr("SUM(o.points_points)")]);

        return (int)$conn->query($query)->fetchColumn();
    }

    protected function _beforeSave() {
        parent::_beforeSave();

        $this->setData("updated_at", date("Y-m-d\\TG:i:s"));

        return $this;
    }
}

