#!/usr/bin/env bash

# TODO:
# * Deduplicate container boot/run logic

# Error on command errors, missing variables, and make pipes fail if any of the involved commands fail
set -euo pipefail
# Required to make *.sql glob return empty array instead of literal string
shopt -s nullglob

DEBUG=
# Missing in Bash
GID=$(id -g)
SCRIPT_NAME=$(basename "$0")
PROJECT_NAME=project-$(basename "$PWD")
MAGE_RUN_CODE=${MAGE_RUN_CODE:-magento}

TEST_IMAGE=$PROJECT_NAME-test
COVERAGE_IMAGE=$PROJECT_NAME-coverage
FPM_IMAGE=$PROJECT_NAME-fpm

IMAGES=("$TEST_IMAGE" "$COVERAGE_IMAGE" "$FPM_IMAGE")

DB_CONTAINER=$PROJECT_NAME-mariadb
FPM_CONTAINER=$PROJECT_NAME-fpm
NGINX_CONTAINER=$PROJECT_NAME-nginx

NGINX_PORT=${NGINX_PORT:-8080}
NETWORK=$PROJECT_NAME

DATABASE_NAME=testing
DATABASE_VOLUMES=()

for f in "$PWD"/test/*.sql; do
	DATABASE_VOLUMES+=(-v "$f:/docker-entrypoint-initdb.d/$(basename "$f"):ro")
done

# Quicker health check since it is a test
# health check must use IP to avoid socket since socket is ready before TCP/IP
DATABASE_PARAMS=(
	-e MYSQL_ALLOW_EMPTY_PASSWORD=yes
	-e MYSQL_DATABASE="$DATABASE_NAME"
	--network="$NETWORK"
	--health-cmd='mysql -h 127.0.0.1 -e "show databases;"'
	--health-interval=2s
	"${DATABASE_VOLUMES[@]}" mariadb
)

# Use fcgi to check if it is alive, should return status once running if
# pm.status_path = /status is set properly
FPM_PARAMS=(
# We need to set UID and GID to allow is_writable to work
	-u "$UID:$GID"
	--network="$NETWORK"
	-v "$PWD:/var/www"
	--health-interval=2s
	"$FPM_IMAGE"
)

NGINX_PARAMS=(
	--network="$NETWORK"
	-e PHP_FPM_HOST="$FPM_CONTAINER"
	-e MAGE_RUN_CODE="$MAGE_RUN_CODE"
	-v "$PWD:/var/www:ro"
	-v "$PWD/dev/nginx:/etc/nginx/templates:ro"
	-p "$NGINX_PORT:80"
	--health-cmd="curl -Iq http://localhost:80"
	--health-interval=5s
	nginx:alpine
)

# Echos parameters if debug is enabled
debug_echo() {
	if [ -n "$DEBUG" ]; then
		echo "$@"
	fi
}

debug_run_cmd() {
	local first=""

	if [ -n "$DEBUG" ]; then
		for var in "$@"; do
			printf "%s'%s'" "$first" "$var"
			first=" "
		done

		echo ""
	fi

	"$@"
}

create_image() {
	local image

	case "$1" in
		"$TEST_IMAGE") image=php-cli-test;;
		"$COVERAGE_IMAGE") image=php-cli-coverage;;
		"$FPM_IMAGE") image=php-fpm;;
		*) echo "Unknown image $1"; exit 1;;
	esac

	if docker image inspect "$1" >/dev/null 2>&1; then
		debug_echo "Reusing $1 image"
	else
		# Pipe Dockerfile to avoid context
		debug_run_cmd docker build --target $image -t "$1" - < ./Dockerfile
	fi
}

is_container_running() {
	[ "$( docker container inspect -f '{{.State.Running}}' "$1" 2>/dev/null )" == "true" ]
}

start_container() {
	debug_run_cmd docker run -d --rm --name "$@"

	while [ "$(docker inspect --format "{{.State.Health.Status }}" "$1")" != "healthy" ]; do
		printf "."
		sleep 1
	done

	echo ""
}

sub_help() {
	echo "Usage: $SCRIPT_NAME [-v] <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    composer   Run composer"
	echo "    test       Run tests"
	echo "    psalm      Run psalm"
	echo "    phpunit    Run phpunit"
	echo "    coverage   Run phpunit and record code coverage"
	echo "    syntax     Run PHP syntax check"
	echo "    network    Manage the test network"
	echo "    database   Manage the test database container"
	echo "    fpm        Manage the test PHP-FPM container"
	echo "    nginx      Manage the test Nginx container"
	echo "    stop       Stops all containers, removes all networks"
	echo "               and clears the cache"
	echo "    clean      Removes containers, images, networks, unused volumes"
	echo "    distclean  Removes containers, images, networks, unused volumes,"
	echo "               coverage, vendor, and magento folders, composer.lock"
	echo ""
	echo "Options:"
	echo "    -v --verbose  Activates verbose print of containers"
	echo ""
	echo "For help with each subcommand run:"
	echo "$SCRIPT_NAME <subcommand> -h|--help"
	echo ""
}

sub_syntax() {
	case "${1:-}" in
		"-h" | "--help")
			echo "Usage: $SCRIPT_NAME [-a]"
			echo ""
			echo "Options:"
			echo "    -a --all  Runs PHP syntax check on all .php files in the"
			echo "              current folder including files outside source control"
			echo ""

			;;
		"-a" | "--all")
			create_image "$TEST_IMAGE"

			# --line-buffered is used to produce immediate output
			# -i only since we pipe input but do not need a TTY
			# && exit 1 || [ "$?" -eq "1" ] is used to produce ok exit code on no
			# output, and error on output
			debug_run_cmd find . -type f -name '*.php' | docker run --rm --init -i \
				-v "$PWD:/app:ro" -w /app \
				"$TEST_IMAGE" sh -c "xargs -i sh -c \" php -l '{}' || true\"" | \
				grep -vE --line-buffered '^No syntax errors|^Errors parsing|^[[:space:]]*$' && \
				exit 1 || [ "$?" -eq "1" ]
			;;
		*)
			create_image "$TEST_IMAGE"

			debug_run_cmd git ls-files '**.php' | docker run --rm --init -i \
				-v "$PWD:/app:ro" -w /app \
				"$TEST_IMAGE" sh -c "xargs -i sh -c \"php -l {} || true\"" | \
				grep -vE --line-buffered '^No syntax errors|^Errors parsing|^[[:space:]]*$' && \
				exit 1 || [ "$?" -eq "1" ]
			;;
	esac
}

sub_psalm() {
	create_image "$TEST_IMAGE"

	# --init for an init wrapper to manage INT
	debug_run_cmd docker run --rm --init -it -v "$PWD:/app" -w /app "$TEST_IMAGE" php vendor/bin/psalm "$@"
}

sub_phpunit() {
	debug_run_cmd rm -rf "magento/var/cache/*"

	create_image "$TEST_IMAGE"
	database_up

	debug_run_cmd docker run --rm -it --init \
		-v "$PWD:/app" -w /app \
		--network="$NETWORK" \
		-e DB_HOST="$DB_CONTAINER" \
		-e DB_NAME="$DATABASE_NAME" \
		"$TEST_IMAGE" \
		php vendor/bin/phpunit "$@"
}

sub_coverage() {
	debug_run_cmd rm -rf "magento/var/cache/*"

	create_image "$COVERAGE_IMAGE"
	database_up

	debug_run_cmd docker run --rm -it --init \
		-v "$PWD:/app" -w /app \
		--network="$NETWORK" \
		-e DB_HOST="$DB_CONTAINER" \
		"$COVERAGE_IMAGE" \
		php -d pcov.enabled=1 -d memory_limit=2048M vendor/bin/phpunit --coverage-html coverage
}

sub_test() {
	sub_psalm
	sub_phpunit
	sub_syntax
}

sub_stop() {
	nginx_down
	fpm_down
	database_down
	network_down

	debug_run_cmd rm -rf "magento/var/cache/*"
}

sub_clean() {
	sub_stop

	debug_run_cmd docker image rm -f "${IMAGES[@]}" 2> /dev/null || true

	for v in $(docker volume ls -qf dangling=true); do
		debug_run_cmd docker volume rm "$v"
	done
}

sub_database() {
	run "database_" "$@"
}

database_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    info  Shows information about database host and port"
	echo "    query Opens up a mysql query prompt"
	echo "    up    Starts the database container for testing"
	echo "    down  Stops the database container for testing"
	echo "    run   Runs the database container interactively, useful"
	echo "          for debugging purposes"
	echo "    log   Shows the logs for database container for testing"
	echo ""
}

database_info() {
	echo "mysql://$DB_CONTAINER:3306/$DATABASE_NAME"
}

database_query() {
	network_up
	database_up

	debug_run_cmd docker exec -it "$DB_CONTAINER" mysql -D "$DATABASE_NAME"
}

database_run() {
	network_up

	debug_run_cmd docker run -it --rm --name "$DB_CONTAINER" "${DATABASE_PARAMS[@]}"
}

database_up() {
	if is_container_running "$DB_CONTAINER"; then
		debug_echo "Reusing running $DB_CONTAINER container"

		return
	fi

	network_up

	start_container "$DB_CONTAINER" "${DATABASE_PARAMS[@]}"
}

database_log() {
	debug_run_cmd docker logs "$DB_CONTAINER" "$@"
}

database_down() {
	debug_run_cmd docker rm -f "$DB_CONTAINER" 2>/dev/null || true
}

database_allow-symlinks() {
	database_up

	debug_run_cmd docker exec -it "$DB_CONTAINER" \
		mysql -D "$DATABASE_NAME" \
		-e "INSERT INTO core_config_data (scope, scope_id, path, value) VALUES ('default', 0, 'dev/template/allow_symlink', '1') ON DUPLICATE KEY UPDATE value='1';"
}

sub_network() {
	run "network_" "$@"
}

network_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    up   Starts the network interface"
	echo "    down Stops the network interface"
	echo ""
}

network_up() {
	if docker network inspect -f '{{.Name}}' "$NETWORK" >/dev/null 2>&1; then
		debug_echo "Reusing network $NETWORK"

		return
	fi

	debug_run_cmd docker network create -d bridge "$NETWORK" >/dev/null
}

network_down() {
	debug_run_cmd docker network rm "$NETWORK" 2>/dev/null || true
}

sub_fpm() {
	run "fpm_" "$@"
}

fpm_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    up   Starts the PHP FPM container"
	echo "    down Stops the PHP FPM container"
	echo "    run  Runs the PHP FPM container interactively, useful"
	echo "         for debugging purposes"
	echo "    log  Shows the logs for the PHP FPM container"
	echo ""
}

fpm_run() {
	create_image "$FPM_IMAGE"
	database_up
	network_up

	debug_run_cmd rm -rf "magento/var/cache/*"

	debug_run_cmd docker run -it --init --rm --name "$FPM_CONTAINER" "${FPM_PARAMS[@]}"
}

fpm_up() {
	if is_container_running "$FPM_CONTAINER"; then
		debug_echo "Reusing running $FPM_CONTAINER container"

		return
	fi

	create_image "$FPM_IMAGE"
	database_up
	network_up

	debug_run_cmd rm -rf "magento/var/cache/*"

	start_container "$FPM_CONTAINER" "${FPM_PARAMS[@]}"
}

fpm_down() {
	docker rm -f "$FPM_CONTAINER" 2>/dev/null || true
}

fpm_log() {
	docker logs "$FPM_CONTAINER" "$@"
}

sub_nginx() {
	run "nginx_" "$@"
}

nginx_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    up   Starts the Nginx container"
	echo "    down Stops the Nginx container"
	echo "    log  Shows the logs for the Nginx container"
	echo ""
}

nginx_run() {
	network_up
	fpm_up

	debug_run_cmd docker run --rm -it --name "$NGINX_CONTAINER" "${NGINX_PARAMS[@]}"
}

nginx_up() {
	if is_container_running "$NGINX_CONTAINER"; then
		debug_echo "Reusing running $NGINX_CONTAINER container"

		return
	fi

	network_up
	fpm_up

	start_container "$NGINX_CONTAINER" "${NGINX_PARAMS[@]}"
}

nginx_down() {
	docker rm -f "$NGINX_CONTAINER" 2>/dev/null || true
}

nginx_log() {
	docker logs "$NGINX_CONTAINER" "$@"
}

sub_composer() {
	local COMPOSER_HOME=${COMPOSER_HOME:-$HOME/.config/composer}

	# --rm to delete
	# -it for interactive tty
	# -u for correct user/group to create files correctly
	# .ssh, passwd, and group volume to be able to use git in composer
	docker run --rm -it -u "$UID:$GID" \
		-v "$PWD:/app" -v "$COMPOSER_HOME:/tmp" \
		-v "$HOME/.ssh:/home/$USER/.ssh" -v /etc/passwd:/etc/passwd:ro -v /etc/group:/etc/group:ro \
		composer:1 "$@"
}

sub_distclean() {
	sub_clean

	rm -rf coverage vendor html magento composer.lock
}

run() {
	local prefix=$1
	local subcommand="${2:-}"

	case "$subcommand" in
		"" | "-h" | "--help")
			"${prefix}help"
			;;

		"-v" | "--verbose")
			DEBUG="true"

			shift
			shift

			run "$prefix" "$@"

			;;

		*)
			shift
			shift

			SCRIPT_NAME="$SCRIPT_NAME $subcommand"

			if declare -f "$prefix$subcommand" > /dev/null; then
				"$prefix$subcommand" "$@"
			else
				echo "Error: '$SCRIPT_NAME' is not a known subcommand." >&2
				"${prefix}help"
				exit 1
			fi
			;;
	esac
}

run "sub_" "$@"