<?php

declare(strict_types=1);

namespace Points\Core;

use function Fixtures\loadPoints;

use Exception;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Customer;
use Mage_Customer_Model_Group;
use Mage_Sales_Model_Quote;
use Points_Core_Helper_Data;
use Throwable;
use Varien_Object;
use Varien_Profiler;

use Points\Core\ProviderInterface;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Fixtures\Customer;
use PHPUnit\Framework\TestCase;
use Points\Core\Extension\Quote;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/../Fixtures/Points.php";

class QuoteTest extends TestCase {
    use MatchesSnapshots;

    public static function setUpBeforeClass(): void {
        loadPoints();
    }

    public function setUp(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->clearQuotes();

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    protected function clearQuotes(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        try {
            $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

            $setupModel->deleteConfigData("carriers/freeshipping/active", "websites");
            $setupModel->deleteConfigData("carriers/freeshipping/free_shipping_subtotal", "websites");

            // Prevent any existing quotes from being loaded
            $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

            if($conn) {
                $conn->query("UPDATE sales_flat_quote SET is_active = 0");
            }
        }
        catch(Exception $e) {
            error_log((string)$e);
        }

        MagentoManager::reset();
    }

    public function loginCustomer(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $customer = Mage::getModel("customer/customer");
        $session = Mage::getSingleton("customer/session");

        $customer->setWebsiteId($store->getWebsiteId());
        $customer->loadByEmail("test-customer@example.com");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
    }

    public function createQuote(): Mage_Sales_Model_Quote {
        // Empty mock required since we have the config
        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
                ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 2,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return $quote;
    }

    public function testSetPointsEmpty(): void {
        $this->loginCustomer();

        $quote = Mage::getModel("sales/quote");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote->setCustomer(Mage::getSingleton("customer/session")->getCustomer());
        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");

        // reload quote
        $checkoutSession->setQuoteId($quote->getId());

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsNotAllowed(): void {
        $this->loginCustomer();

        $quote = $this->createQuote();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");

        // reload quote
        $checkoutSession->setQuoteId($quote->getId());

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(false);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsNoCustomer(): void {
        MagentoManager::init();

        $quote = $this->createQuote();

        MagentoManager::reset();
        MagentoManager::init();

        $checkoutSession = Mage::getSingleton("checkout/session");

        // reload quote
        $checkoutSession->setQuoteId($quote->getId());

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPoints(): void {
        $this->loginCustomer();
        $this->createQuote();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsMaximum(): void {
        $this->loginCustomer();
        $this->createQuote();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        points { incVat exVat vat }
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    maximumAvailable { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                    shippingValue { incVat exVat vat }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsSimpleMaximum(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        points { incVat exVat vat }
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    maximumAvailable { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                    shippingValue { incVat exVat vat }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsSimpleOne(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        points { incVat exVat vat }
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    maximumAvailable { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                    shippingValue { incVat exVat vat }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 1, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsSimpleZero(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        points { incVat exVat vat }
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    maximumAvailable { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                    shippingValue { incVat exVat vat }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 0, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsSimpleMaximumLimited(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        points { incVat exVat vat }
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    maximumAvailable { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                    shippingValue { incVat exVat vat }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }


    public function testSetPointsSimpleMaximumLimitedNotIncludingTax(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        points { incVat exVat vat }
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    maximumAvailable { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                    shippingValue { incVat exVat vat }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(false);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsVirtualMaximum(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(99999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        points { incVat exVat vat }
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumCurrency { incVat exVat vat }
                    }
                }
                availablePointPayments {
                    id
                    label
                    minimum { incVat exVat vat }
                    maximum { incVat exVat vat }
                    maximumAvailable { incVat exVat vat }
                    quoteValue { incVat exVat vat }
                    shippingValue { incVat exVat vat }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPayment(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsToMaximumSimpleMaximum(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPaymentToMaximum(id: "TEST") {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsToMaximumSimpleMaximumLimited(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPaymentToMaximum(id: "TEST") {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetPointsToMaximumSimpleMaximumLimitedExVat(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(false);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            setQuotePointsPaymentToMaximum(id: "TEST") {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            minimum { incVat exVat vat }
                            maximum { incVat exVat vat }
                            maximumCurrency { incVat exVat vat }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                    }
                    selectedPointPayment {
                        id
                        label
                        minimum { incVat exVat vat }
                        maximum { incVat exVat vat }
                        maximumAvailable { incVat exVat vat }
                        quoteValue { incVat exVat vat }
                        shippingValue { incVat exVat vat }
                        points { incVat exVat vat }
                        pointsInCurrency { incVat exVat vat }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testPlaceOrder(): void {
        MagentoManager::initAdmin();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData("carriers/freeshipping/active", 1, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        /**
         * @var Quote $quote
         */
        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(9999);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->save();

        $quote->getPayment()->importData([
            "method" => "free",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->method("getUpdatedCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceEnd")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->never())
            ->method("onQuoteSubmitFailure");
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitAfter")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
                ...on PlaceOrderResultSuccess {
                    order {
                        status
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            pointsInCurrency { incVat exVat vat }
                        }
                        grandTotal { incVat exVat vat }
                        subTotal { incVat exVat vat }
                        discountTotal
                        shipping {
                            method {
                                code
                                description
                            }
                            total {
                                incVat
                                exVat
                                vat
                            }
                        }
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
