<?php

declare(strict_types=1);

namespace Points\Core;

use Mage;
use Mage_Sales_Model_Quote_Address;
use Mage_Sales_Model_Quote_Item_Abstract;
use Mage_Tax_Model_Config;

class QuoteAddressTotal extends Total {
    const ROW_SHIPPING = "shipping";

    /**
     * @var Array<string, Mage_Sales_Model_Quote_Item_Abstract>
     */
    protected $indices;

    /**
     * @param Array<string, Mage_Sales_Model_Quote_Item_Abstract> $indices
     * @param array{min:Array<string, Points>, max:Array<string, Points>, value:Array<string, Points>} $points
     * @param Array<Currency> $currency
     */
    protected function __construct(
        array $indices,
        array $points,
        array $currency
    ) {
        $this->indices = $indices;

        parent::__construct($currency, $points["min"], $points["max"], $points["value"]);
    }

    public function getRowPointsValue(Mage_Sales_Model_Quote_Item_Abstract $item): ?Points {
        $rowId = array_search($item, $this->indices, true);

        if($rowId === false) {
            return null;
        }

        return $this->points[$rowId] ?? null;
    }

    public function getRowPointsMin(Mage_Sales_Model_Quote_Item_Abstract $item): ?Points {
        $rowId = array_search($item, $this->indices, true);

        if($rowId === false) {
            return null;
        }

        return $this->minPoints[$rowId] ?? null;
    }

    public function getRowPointsMax(Mage_Sales_Model_Quote_Item_Abstract $item): ?Points {
        $rowId = array_search($item, $this->indices, true);

        if($rowId === false) {
            return null;
        }

        return $this->maxPoints[$rowId] ?? null;
    }

    public function getRowCurrency(Mage_Sales_Model_Quote_Item_Abstract $item): ?Currency {
        $rowId = array_search($item, $this->indices, true);

        if($rowId === false) {
            return null;
        }

        return $this->currency[$rowId] ?? null;
    }

    public function getShippingPointsValue(): ?Points {
        return $this->points[self::ROW_SHIPPING] ?? null;
    }

    public function getShippingPointsMin(): ?Points {
        return $this->minPoints[self::ROW_SHIPPING] ?? null;
    }

    public function getShippingPointsMax(): ?Points {
        return $this->maxPoints[self::ROW_SHIPPING] ?? null;
    }

    public function getShippingCurrency(): ?Currency {
        return $this->currency[self::ROW_SHIPPING] ?? null;
    }

    /**
     * @return array{currency: Currency, points:Points, minPoints:?Points, maxPoints:?Points}
     */
    public function getRowItemData(Mage_Sales_Model_Quote_Item_Abstract $item): ?array {
        $rowId = null;

        foreach($this->indices as $k => $i) {
            if($i->getId() == $item->getId()) {
                $rowId = $k;

                break;
            }
        }

        if( ! $rowId) {
            return null;
        }

        if( ! array_key_exists($rowId, $this->currency) || ! array_key_exists($rowId, $this->points)) {
            return null;
        }

        return [
            "currency" => $this->currency[$rowId],
            "points" => $this->points[$rowId],
            "minPoints" => $this->minPoints[$rowId] ?? null,
            "maxPoints" => $this->maxPoints[$rowId] ?? null,
        ];
    }

    public static function fromQuoteAddress(
        Mage_Sales_Model_Quote_Address $address,
        string $type,
        ProviderInterface $provider
    ): self {
        // $helper = Mage::helper("points_core");
        $items = $address->getAllItems();
        $quote = $address->getQuote();
        $customerGroupId = $quote->getCustomerGroupId();
        $store = $quote->getStore();
        $productIds = [];
        $indices = [];

        foreach($items as $k => $i) {
            $productIds[] = $i->getProductId();

            // Only consider parents, since children do not have any amounts
            if($i->getParentItemId()) {
                continue;
            }

            // Kinda ugly way of keeping track of item indices
            // TODO: Improve
            $indices["idx_$k"] = $i;

            // FIXME: Handle Bundle
        }

        $resource = Mage::getResourceModel("points_core/product_price");
        $pricesIncludeTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $shippingIncludesTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX);

        $prices = $resource->getMatchingPrices($store, $customerGroupId, $type, $productIds);
        /**
         * @var Array<string, Currency> $currency
         */
        $currency = [];
        /**
         * @var array{min: Array<string, Amount<float>>, max: Array<string, Amount<float>>, value: Array<string, Amount<float>>} $points
         */
        $points = [
            "min" => [],
            "max" => [],
            "value" => [],
        ];

        foreach($indices as $id => $item) {
            $included = false;
            $productId = $item->getProductId();

            if($item->getProductType() === "configurable") {
                // We have a configurable item, those store their price on the
                // child product but report the price on the parent row, we need
                // to use the child item product id to find the correct price:

                foreach($items as $i) {
                    if($i->getParentItemId() == $item->getId()) {
                        $productId = $i->getProductId();

                        break;
                    }
                }
            }

            $rowBeforeDiscount = $item->getBaseRowTotal() + $item->getBaseTaxAmount() + $item->getBaseHiddenTaxAmount();
            $discountRate = $rowBeforeDiscount > 0 ? $item->getBaseDiscountAmount() / $rowBeforeDiscount : 0;
            $nonDiscountRate = 1 - $discountRate;

            if( ! empty($prices[$productId])) {
                $included = true;
                $p = $prices[$productId];

                $val = $nonDiscountRate * $item->getQty() * (float)$p->getPrice();
                $max = $nonDiscountRate * $item->getQty() * (float)($p->getMaxPrice() ? $p->getMaxPrice() : $p->getPrice());
                $min = $item->getQty() * (float)$p->getMinPrice();

                $taxRate = $item->getTaxPercent() / 100;
                $taxConversionRate = $pricesIncludeTax ? $taxRate / (1 + $taxRate) : $taxRate;

                $points["min"][$id] = new Amount($min, $pricesIncludeTax, $min * $taxConversionRate);
                $points["max"][$id] = new Amount($max, $pricesIncludeTax, $max * $taxConversionRate);
                $points["value"][$id] = new Amount($val, $pricesIncludeTax, $val * $taxConversionRate);
            }

            $currency[$id] = new Currency(
                $pricesIncludeTax ?
                    $item->getBaseRowTotalInclTax() - $item->getBaseDiscountAmount() :
                    $item->getBaseRowTotal() - $item->getBaseDiscountAmount() + $item->getBaseHiddenTaxAmount(),
                $pricesIncludeTax,
                (float)$item->getBaseTaxAmount(),
                $included
            );
        }

        if($address->getShippingAmount() > 0) {
            $included = false;
            $amount = $provider->getQuoteShippingPrice($address);

            if($amount !== null) {
                $included = true;
                $taxRate = ($address->getBaseShippingTaxAmount() + $address->getBaseShippingHiddenTaxAmount())
                    / $address->getBaseShippingAmount();
                $taxConversionRate = $shippingIncludesTax ? $taxRate / (1 + $taxRate) : $taxRate;

                $points["value"][self::ROW_SHIPPING] = new Amount($amount, $shippingIncludesTax, $amount * $taxConversionRate);
                // TODO: Configurable minimum shipping amount?
                $points["min"][self::ROW_SHIPPING] = new Amount(0, $shippingIncludesTax, 0);
                $points["max"][self::ROW_SHIPPING] = new Amount($amount, $shippingIncludesTax, $amount * $taxConversionRate);
            }

            $currency["shipping"] = new Currency(
                $shippingIncludesTax ?
                    $address->getBaseShippingInclTax() - $address->getBaseShippingDiscountAmount() :
                    $address->getBaseShippingAmount() + $address->getBaseShippingHiddenTaxAmount(),
                $shippingIncludesTax,
                (float)$address->getBaseShippingTaxAmount(),
                $included
            );
        }

        $spreadPoints = array_map("Points\\Core\\spreadPoints", $points);

        // TODO: Limit based on max order sum and max totals

        return new self($indices, $spreadPoints, $currency);
    }
}
