<?php

declare(strict_types=1);

use Points\Core\Extension\Quote;
use Points\Core\Points;
use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;
use Points\Core\Total;

class Points_Core_Model_Schema_QuotePointPaymentMethod {
    /**
     * @var string
     */
    protected $type;

    /**
     * @var Quote
     */
    protected $quote;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @var ?Array<QuoteAddressTotal>
     */
    protected $totals;

    public function __construct(
        string $type,
        Mage_Sales_Model_Quote $quote,
        ProviderInterface $provider
    ) {
        $this->type = $type;
        /**
         * @var Quote
         */
        $this->quote = $quote;
        $this->provider = $provider;
    }

    public function getId(): string {
        return $this->type;
    }

    public function getLabel(): string {
        return $this->provider->getLabel($this->quote->getStore());
    }

    /**
     * @return Array<QuoteAddressTotal>
     */
    public function getTotals() {
        if($this->totals === null) {
            $this->totals = [];

            foreach($this->quote->getAddressesCollection() as $addr) {
                $this->totals[] = QuoteAddressTotal::fromQuoteAddress($addr, $this->type, $this->provider);
            }
        }

        return $this->totals;
    }

    public function getMinimum(): Points_Core_Model_Schema_QuoteTotal {
        return new Points_Core_Model_Schema_QuoteTotal(
            $this->getTotals(),
            function(Total $t): array {
                return $t->getPointsMin();
            }
        );
    }

    public function getMaximum(): Points_Core_Model_Schema_QuoteTotal {
        return new Points_Core_Model_Schema_QuoteTotal(
            $this->getTotals(),
            function(Total $t): array {
                return $t->getPointsMax();
            }
        );
    }

    public function getMaximumAvailable(): Points_Core_Model_Schema_QuoteTotal {
        $balance = $this->provider->getCustomerPointsBalance($this->quote->getStore(), $this->quote->getCustomer());
        $balanceInclTax = $this->provider->getCustomerPointsBalanceIncludesTax($this->quote->getStore(), $this->quote->getCustomer());
        $quoteValue = $this->getQuoteValue();
        $quoteValueExVat = $quoteValue->getExVat();
        $taxRate = $quoteValueExVat > 0 ? $quoteValue->getVat() / $quoteValueExVat : 0;
        $tax = $taxRate > 0 ?
            ($balanceInclTax ? (int)ceil($balance * $taxRate / (1 + $taxRate)) : (int)floor($balance * $taxRate)) :
            0;

        $available = new Points($balance, $balanceInclTax, $tax);

        return new Points_Core_Model_Schema_QuoteTotalAvailable(
            $this->getTotals(),
            function(Total $t): array {
                return $t->getPointsMax();
            },
            $available
        );
    }

    public function getQuoteValue(): Points_Core_Model_Schema_QuoteTotal {
        return new Points_Core_Model_Schema_QuoteTotal(
            $this->getTotals(),
            function(Total $t): array {
                return $t->getPointsValue();
            }
        );
    }

    public function getShippingValue(): ?Points_Core_Model_Schema_QuoteTotal {
        $shipping = array_filter(array_map(
            function(Total $t): ?Points {
                return $t->getShippingPointsValue();
            },
            $this->getTotals()
        ));

        if(empty($shipping)) {
            return null;
        }

        return new Points_Core_Model_Schema_QuoteTotal(
            $this->getTotals(),
            function(Total $t): array {
                $val = $t instanceof QuoteAddressTotal ? $t->getShippingPointsValue() : null;

                return $val ? [$val] : [];
            }
        );
    }
}
