<?php

declare(strict_types=1);

use GraphQL\Deferred;
use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use Points\Core\Extension\Order;
use Points\Core\Extension\Quote;
use Points\Core\Extension\QuoteAddress;
use Points\Core\QuoteAddressTotal;
use Points\Core\Total;
use Points\Core\ProviderInterface;

class Points_Core_Model_Schema extends MageQL_Core_Model_Schema_Abstract {
    const SUCCESS = "success";
    const ERROR_INVALID_POINT_TYPE = "errorInvalidPointType";
    const ERROR_POINT_TYPE_DOES_NOT_APPLY = "errorPointTypeDoesNotApply";
    const ERROR_POINT_REDEMPTION_NOT_ALLOWED = "errorPointRedemptionNotAllowed";
    const ERROR_CUSTOMER_NOT_LOGGED_IN = "errorCustomerNotLoggedIn";

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "CustomerPointsBalance":
            return $this->object("A customer points balance")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointsSpendingLimit":
            return $this->object("A point spending limit for a time window")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "PointTotal":
            return $this->object("A total in points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "ProductPointPaymentPrice":
            return $this->object("A product price in points")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "QuoteItemPointPayment":
            return $this->object("The point amount for the quote item row for the selected payment method")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "QuotePointPaymentMethod":
            return $this->object("An available point payment method")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SelectedOrderPointPaymentMethod":
            return $this->object("Information about points spent on an order")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SelectedQuotePointPaymentMethod":
            return $this->object("The selected payment method")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SetQuotePointsPaymentResult":
            return $this->object("Result of setQuotePointsPayment")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "SetQuotePointsPaymentResultType":
            return $this->enum("Result type of setQuotePointsPayment", [
                self::SUCCESS => [
                    "description" => "Successfully set the points",
                ],
                self::ERROR_INVALID_POINT_TYPE => [
                    "description" => "The supplied point type is invalid",
                ],
                self::ERROR_POINT_TYPE_DOES_NOT_APPLY => [
                    "description" => "The supplied point type does not apply to the quote",
                ],
                self::ERROR_POINT_REDEMPTION_NOT_ALLOWED => [
                    "description" => "The currently logged in customer is not allowed to redeem points of this type",
                ],
                self::ERROR_CUSTOMER_NOT_LOGGED_IN => [
                    "description" => "Customer must be logged in to be able to redeem points",
                ],
            ]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "ConfigurationOptionItem":
            return [
                "pointsPrices" => $this->field("[ProductPointPaymentPrice!]!", "")
                    ->setResolver(function(
                        MageQL_Catalog_Model_Product_Configurable_Option $option,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ): Deferred {
                        $customerGroupId = (int)Mage::getSingleton("customer/session")->getCustomerGroupId();
                        $collector = Points_Core_Model_Product_Price_Collector::instance($ctx->getStore(), $customerGroupId);
                        $product = $option->getProduct();

                        $collector->queue($product);

                        return new Deferred(function() use($product, $collector) {
                            return $collector->getPrices($product);
                        });
                    }),
            ];

        case "Customer":
            return [
                "points" => $this->field("[CustomerPointsBalance!]!", "List of point currencies which are available for the customer")
                    ->setResolver(function(
                        Mage_Customer_Model_Customer $customer,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ): array {
                        $helper = Mage::helper("points_core");
                        $providers = $helper->getTypeProviders($ctx->getStore());

                        $enabledProviders = array_filter($providers, function($p) use($ctx) {
                            return $p->isEnabled($ctx->getStore());
                        });

                        return array_map(function(
                            string $key,
                            ProviderInterface $provider
                        ) use($customer): Points_Core_Model_Schema_CustomerPointsBalance {
                            return new Points_Core_Model_Schema_CustomerPointsBalance($key, $provider, $customer);
                        }, array_keys($enabledProviders), $enabledProviders);
                    }),
            ];

        case "CustomerPointsBalance":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String!", "Label of the point currency type"),
                "redemptionAllowed" => $this->field("Boolean!", "If this customer is allowed to redeem this point currency"),
                "points" => $this->field("Int!", "Total price in points for this product if using this payment method"),
                "spendingLimit" => $this->field("PointsSpendingLimit", "Limit to the number of points spent in a time window if any"),
            ];

        case "PointsSpendingLimit":
            return [
                "spent" => $this->field("Int!", "Amount of points spent in the current time window"),
                "limit" => $this->field("Int!", "Amount of points spent in the current time window"),
                "remaining" => $this->field("Int!", "Amount of points spent in the current time window"),
                "resetsAt" => $this->field("String!", "Datetime when the current time window ends"),
            ];

        case "ListProduct":
        case "ProductDetail":
            return [
                "pointsPrices" => $this->field("[ProductPointPaymentPrice!]!", "List of available point payment options for this product")
                    ->setResolver(function(
                        Mage_Catalog_Model_Product $product,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ): Deferred {
                        $customerGroupId = (int)Mage::getSingleton("customer/session")->getCustomerGroupId();
                        $collector = Points_Core_Model_Product_Price_Collector::instance($ctx->getStore(), $customerGroupId);

                        $collector->queue($product);

                        return new Deferred(function() use($product, $collector) {
                            return $collector->getPrices($product);
                        });
                    }),
            ];

        case "ProductPointPaymentPrice":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String!", "Label of the point currency type"),
                "minimum" => $this->field("ProductPrice!", "Minimum amount of points to possible spend on this product if using this payment method"),
                "maximum" => $this->field("ProductPrice!", "Maximum amount of points to possible spend on this product if using this payment method"),
                "price" => $this->field("ProductPrice!", "Total price in points for this product if using this payment method"),
                "maximumCurrency" => $this->field("ProductPrice!", "Maximum price in currency for this product if using this payment method, accounting for minimum number of points spent"),
            ];

        case "Mutation":
            return [
                "setQuotePointsPayment" => $this->field("SetQuotePointsPaymentResult!", "Attempts to set a certain number of points of a specific type to spend on this quote")
                    ->addArgument("id", $this->argument("ID!", "Point type identifier"))
                    ->addArgument("points", $this->argument("Int!", "Amount of points to spend on this quote"))
                    ->addArgument("incVat", $this->argument("Boolean!", "If the amount of points to spend includes VAT"))
                    ->setResolver(
                        /**
                         * @param mixed $unusedSrc
                         * @param array{id:string, points:int, incVat:bool} $args
                         */
                        function($unusedSrc, array $args): Varien_Object {
                            return $this->setPoints($args["id"], function(Mage_Sales_Model_Quote $quote) use($args): void {
                                /**
                                 * @var Quote $quote
                                 */

                                $quote->setPointsWanted($args["points"]);
                                $quote->setPointsWantedIncludesTax($args["incVat"]);
                            });
                        }
                ),
                "setQuotePointsPaymentToMaximum" => $this->field("SetQuotePointsPaymentResult!", "Attempts to set the maximum number of points available to spend for a specific type of points on the current quote")
                    ->addArgument("id", $this->argument("ID!", "Point type identifier"))
                    ->setResolver(
                        /**
                         * @param mixed $unusedSrc
                         * @param array{id:string} $args
                         */
                        function($unusedSrc, array $args): Varien_Object {
                            return $this->setPoints(
                                $args["id"],
                                function(
                                    Mage_Sales_Model_Quote $quote,
                                    Mage_Customer_Model_Customer $customer,
                                    ProviderInterface $provider
                                ): void {
                                    /**
                                     * @var Quote $quote
                                     */
                                    $points = $provider->getCustomerPointsBalance($quote->getStore(), $customer);
                                    $incVat = $provider->getCustomerPointsBalanceIncludesTax($quote->getStore(), $customer);

                                    $quote->setPointsWanted($points);
                                    $quote->setPointsWantedIncludesTax($incVat);
                                }
                            );
                        }
                    ),
            ];

        case "PointTotal":
            return [
                "exVat" => $this->field("Int!", "Price excluding VAT"),
                "incVat" => $this->field("Int!", "Price including VAT"),
                "vat" => $this->field("Int!", "VAT amount"),
            ];

        case "Order":
            return [
                "selectedPointPayment" => $this->field("SelectedOrderPointPaymentMethod", "The selected quote point payment if any")
                    ->setResolver(function(
                        Mage_Sales_Model_Order $order,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ) {
                        /**
                         * @var Order $order
                         */
                        $helper = Mage::helper("points_core");
                        $type = $order->getPointsType();

                        if( ! $type) {
                            return null;
                        }

                        $provider = $helper->getTypeProvider($ctx->getStore(), $type);

                        return new Points_Core_Model_Schema_SelectedOrderPointPaymentMethod($type, $order, $provider);
                    }),
            ];

        case "Quote":
            return [
                "availablePointPayments" => $this->field("[QuotePointPaymentMethod!]!", "List of available point payment methods which can be used in addition to the standard payment method")
                    ->setResolver(function(
                        Mage_Sales_Model_Quote $src,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ): array {
                        $helper = Mage::helper("points_core");
                        $providers = $helper->getTypeProviders($ctx->getStore());

                        $appliedProviders = array_filter($providers, function($p) use($src) {
                            return $p->appliesTo($src);
                        });

                        return array_map(function($p, $k) use($src) {
                            return new Points_Core_Model_Schema_QuotePointPaymentMethod($k, $src, $p);
                        }, $appliedProviders, array_keys($appliedProviders));
                    }),
                "selectedPointPayment" => $this->field("SelectedQuotePointPaymentMethod", "The selected quote point payment if any")
                    ->setResolver(function(
                        Mage_Sales_Model_Quote $quote,
                        array $unusedArgs,
                        MageQL_Core_Model_Context $ctx
                    ) {
                        /**
                         * @var Quote $quote
                         */
                        $helper = Mage::helper("points_core");
                        $type = $quote->getPointsType();

                        if( ! $type) {
                            return null;
                        }

                        $provider = $helper->getTypeProvider($ctx->getStore(), $type);

                        if( ! $provider) {
                            return null;
                        }

                        return new Points_Core_Model_Schema_SelectedQuotePointPaymentMethod($type, $quote, $provider);
                    }),
            ];

        case "QuoteItem":
            return [
                "selectedPointPayment" => $this->field("QuoteItemPointPayment", "Information about the currently selected point payment for this quote item, if any is selected and available for this item")
                    ->setResolver(function(
                        Mage_Sales_Model_Quote_Item $item
                    ): ?Points_Core_Model_Schema_QuoteItemPointPayment {
                        $helper = Mage::helper("points_core");
                        /**
                         * @var Quote
                         */
                        $quote = $item->getQuote();

                        $currency = [];
                        $points = [];
                        $minPoints = [];
                        $maxPoints = [];
                        $type = $quote->getPointsType();
                        $store = $quote->getStore();
                        $items = array_merge([$item], $item->getChildren());

                        if( ! $type) {
                            return null;
                        }

                        $provider = $helper->getTypeProvider($store, $type);

                        if( ! $provider) {
                            return null;
                        }

                        // The total will only be present on one item
                        foreach($quote->getAllAddresses() as $address) {
                            /**
                             * @var QuoteAddress $address
                             */
                            $total = $helper->getQuoteAddressTotal($address, $type, $provider);

                            if( ! $total) {
                                continue;
                            }

                            foreach($items as $i) {
                                $tuple = $total->getRowItemData($i);

                                if( ! $tuple) {
                                    continue;
                                }

                                $currency[] = $tuple["currency"];
                                $points[] = $tuple["points"];

                                if($tuple["minPoints"]) {
                                    $minPoints[] = $tuple["minPoints"];
                                }

                                if($tuple["maxPoints"]) {
                                    $maxPoints[] = $tuple["maxPoints"];
                                }
                            }
                        }

                        if(empty($currency) || empty($points)) {
                            return null;
                        }

                        $total = new Total($currency, $minPoints, $maxPoints, $points);

                        return new Points_Core_Model_Schema_QuoteItemPointPayment($type, $provider, $total, $item);
                    }),
            ];

        case "QuoteItemPointPayment":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String!", "Label of the point currency type"),
                "points" => $this->field("PointTotal!", "The amount of points"),
                "minimum" => $this->field("PointTotal!", "The minimum amount of points"),
                "maximum" => $this->field("PointTotal!", "The maximum amount of points"),
                "maximumCurrency" => $this->field("QuoteTotal!", "Maximum amount of currency to spend for this row when using points"),
            ];

        case "QuotePointPaymentMethod":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String!", "Label of the point currency type"),
                "minimum" => $this->field("PointTotal!", "Minimum amount of points to possible spend on this quote if using this payment method"),
                "maximum" => $this->field("PointTotal!", "Maximum amount of points to possible spend on this quote if using this payment method"),
                "maximumAvailable" => $this->field("PointTotal!", "Maximum amount of points available to spend on this quote if using this payment method"),
                "quoteValue" => $this->field("PointTotal!", "Quote value in amount of points for this payment method"),
                "shippingValue" => $this->field("PointTotal", "Point value for shipping, if covered by points"),
                // TODO: Add amount of currency not covered by points
            ];

        case "SelectedOrderPointPaymentMethod":
            return [
                "id" => $this->field("ID!", "ID of the point currency type"),
                "label" => $this->field("String", "Label of the point currency type"),
                "points" => $this->field("PointTotal!", "The amount of points set to spend"),
                "pointsInCurrency" => $this->field("QuoteTotal!", "Points value in local currency"),
            ];

        case "SelectedQuotePointPaymentMethod":
            return array_merge($registry->getFieldBuilders("QuotePointPaymentMethod"), [
                "points" => $this->field("PointTotal!", "The amount of points set to spend"),
                "pointsInCurrency" => $this->field("QuoteTotal!", "Points value in local currency"),
            ]);

        case "SetQuotePointsPaymentResult":
            return [
                "result" => $this->field("SetQuotePointsPaymentResultType!", "Type of result from setQuotePointsPayment"),
                "quote" => $this->field("Quote!", "Current session quote"),
            ];
        }

        return [];
    }

    /**
     * Fetches and validates the point type provider against the currenti quote
     * and customer, then calls setProps to modify the requested point values.
     *
     * It will automatically set points type on the quote before the closure is
     * called, as well as set the totals collected flag to false.
     *
     * @param callable(
     *   Quote,
     *   Mage_Customer_Model_Customer,
     *   ProviderInterface
     * ):void $setProps
     */
    public function setPoints(string $id, $setProps): Varien_Object {
        $helper = Mage::helper("points_core");
        $model = Mage::getSingleton("mageql_sales/quote");
        /**
         * @var Quote $quote
         */
        $quote = $model->getQuote();
        $store = $quote->getStore();
        $customer = Mage::getSingleton("customer/session")->getCustomer();

        $provider = $helper->getTypeProvider($store, $id);

        if( ! $provider) {
            return new Varien_Object([
                "result" => self::ERROR_INVALID_POINT_TYPE,
                "quote" => $quote,
            ]);
        }

        if( ! $provider->appliesTo($quote)) {
            return new Varien_Object([
                "result" => self::ERROR_POINT_TYPE_DOES_NOT_APPLY,
                "quote" => $quote,
            ]);
        }

        if( ! $customer->getId()) {
            return new Varien_Object([
                "result" => self::ERROR_CUSTOMER_NOT_LOGGED_IN,
                "quote" => $quote,
            ]);
        }

        if( ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            return new Varien_Object([
                "result" => self::ERROR_POINT_REDEMPTION_NOT_ALLOWED,
                "quote" => $quote,
            ]);
        }

        $quote->setPointsType($id);

        $setProps($quote, $customer, $provider);

        $quote->setTotalsCollectedFlag(false);

        $model->saveSessionQuote();

        // TODO: Verify it got set
        return new Varien_Object([
            "result" => self::SUCCESS,
            "quote" => $quote,
        ]);
    }
}
