<?php

declare(strict_types=1);

use Points\Core\Extension\Creditmemo;
use Points\Core\Extension\Invoice;
use Points\Core\Extension\Order;
use Points\Core\Extension\Quote;
use Points\Core\Extension\QuoteAddress;
use Points\Core\QuoteAddressTotal;
use Points\Core\ProviderInterface;

class Points_Core_Model_Observer extends Mage_Core_Model_Abstract {
    public function salesQuoteAddressCollectTotalsBefore(Varien_Event_Observer $observer): void {
        /**
         * @var QuoteAddress $address
         */
        $address = $observer->getQuoteAddress();

        $address->unsSelectedPointsTotalInstance();
    }

    public function salesQuoteCollectTotalsBefore(Varien_Event_Observer $observer): void {
        /**
         * @var Quote $quote
         */
        $quote = $observer->getQuote();
        $store = $quote->getStore();
        $pricesIncludeTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);

        if($quote->getPointsType() && $quote->getPointsPoints() && ! $quote->hasPointsWanted()) {
            if($pricesIncludeTax) {
                $quote->setPointsWanted($quote->getPointsPoints() + $quote->getPointsTaxPoints());
                $quote->setPointsWantedIncludesTax(true);
            }
            else {
                $quote->setPointsWanted($quote->getPointsPoints());
                $quote->setPointsWantedIncludesTax(false);
            }
        }

        // TODO: Validate points type
        $quote->setPointsPoints(0);
        $quote->setPointsTaxPoints(0);
        $quote->setBasePointsAmount(0);
        $quote->setBasePointsTaxAmount(0);
        $quote->setPointsAmount(0);
        $quote->setPointsTaxAmount(0);

        $quote->setPointsPointsTotal(0);
        $quote->setPointsTaxPointsTotal(0);
        $quote->setPointsAmountIncluded(0);
        $quote->setPointsTaxAmountIncluded(0);
        $quote->setPointsAmountExcluded(0);
        $quote->setPointsTaxAmountExcluded(0);
    }

    public function salesQuoteCollectTotalsAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();

        if( ! $quote->getPointsType()) {
            return;
        }

        // TODO: Validate points type
        $quote->setPointsPoints(0);
        $quote->setPointsTaxPoints(0);
        $quote->setBasePointsAmount(0);
        $quote->setBasePointsTaxAmount(0);
        $quote->setPointsAmount(0);
        $quote->setPointsTaxAmount(0);

        $quote->setPointsPointsTotal(0);
        $quote->setPointsTaxPointsTotal(0);
        $quote->setPointsAmountIncluded(0);
        $quote->setPointsTaxAmountIncluded(0);
        $quote->setPointsAmountExcluded(0);
        $quote->setPointsTaxAmountExcluded(0);

        $quote->setPointsShippingPointsValue(null);
        $quote->setPointsShippingTaxPointsValue(null);

        /**
         * @var QuoteAddress $addr
         */
        foreach($quote->getAllAddresses() as $addr) {
            $quote->setPointsPoints($quote->getPointsPoints() + (int)$addr->getPointsPoints());
            $quote->setPointsTaxPoints($quote->getPointsTaxPoints() + (int)$addr->getPointsTaxPoints());
            $quote->setBasePointsAmount($quote->getBasePointsAmount() + (float)$addr->getBasePointsAmount());
            $quote->setBasePointsTaxAmount($quote->getBasePointsTaxAmount() + (float)$addr->getBasePointsTaxAmount());
            $quote->setPointsAmount($quote->getPointsAmount() + (float)$addr->getPointsAmount());
            $quote->setPointsTaxAmount($quote->getPointsTaxAmount() + (float)$addr->getPointsTaxAmount());

            $quote->setPointsPointsTotal($quote->getPointsPointsTotal() + (int)$addr->getPointsPointsTotal());
            $quote->setPointsTaxPointsTotal($quote->getPointsTaxPointsTotal() + (int)$addr->getPointsTaxPointsTotal());
            $quote->setPointsAmountIncluded($quote->getPointsAmountIncluded() + (float)$addr->getPointsAmountIncluded());
            $quote->setPointsTaxAmountIncluded($quote->getPointsTaxAmountIncluded() + (float)$addr->getPointsTaxAmountIncluded());
            $quote->setPointsAmountExcluded($quote->getPointsAmountExcluded() + (float)$addr->getPointsAmountExcluded());
            $quote->setPointsTaxAmountExcluded($quote->getPointsTaxAmountExcluded() + (float)$addr->getPointsTaxAmountExcluded());

            if($addr->getPointsShippingPointsValue() !== null) {
                $quote->setPointsShippingPointsValue(
                    ($quote->getPointsShippingPointsValue() ?: 0) + $addr->getPointsShippingPointsValue()
                );
            }

            if($addr->getPointsShippingTaxPointsValue() !== null) {
                $quote->setPointsShippingTaxPointsValue(
                    ($quote->getPointsShippingTaxPointsValue() ?: 0) + $addr->getPointsShippingTaxPointsValue()
                );
            }
        }

        $quote->unsPointsWanted();
        $quote->unsPointsWantedIncludesTax();
    }

    public function salesQuoteSubmitBefore(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();
        /**
         * @var Order
         */
        $order = $observer->getOrder();
        $customer = $quote->getCustomer();
        $store = $quote->getStore();

        // TODO: Verify that we are paying a minimum amount of points if there are some with a non-zero minimum in all points payments
        // TODO: Check order too?

        if(abs($quote->getBasePointsAmount()) < 0.01) {
            return;
        }

        // Ok, we have some amount of spent points, verify

        $provider = $this->getQuoteProvider($quote);
        // Type is always string if we have a provider
        $type = (string)$quote->getPointsType();

        if( ! $provider->appliesTo($quote)) {
            throw new Exception(sprintf(
                "%s: Provider for points type '%s' does not apply to quote %d",
                __METHOD__,
                $type,
                $quote->getId()
            ));
        }

        if( ! $customer->getId()) {
            throw new Exception(sprintf(
                "%s: No customer for quote %d",
                __METHOD__,
                $quote->getId()
            ));
        }

        if( ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            throw new Exception(sprintf(
                "%s: Customer '%d' is not allowed to redeem points of type '%s' for quote %d",
                __METHOD__,
                $customer->getId(),
                $type,
                $quote->getId()
            ));
        }

        // Verify product minimum/maximum
        $totals = array_map(function(Mage_Sales_Model_Quote_Address $a) use($type, $provider): QuoteAddressTotal {
            return QuoteAddressTotal::fromQuoteAddress($a, $type, $provider);
        }, $quote->getAllAddresses());

        // We only check inc-tax, ex-tax should follow
        $pointsMin = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsMin()));
        }, $totals));
        $pointsMax = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsMax()));
        }, $totals));

        if($pointsMin > $quote->getPointsPoints()) {
            throw new Exception(sprintf(
                "%s: Quote '%d' does not have enough points, needs %d, saw %d",
                __METHOD__,
                $quote->getId(),
                $pointsMin,
                $quote->getPointsPoints()
            ));
        }

        if($pointsMax < $quote->getPointsPoints()) {
            throw new Exception(sprintf(
                "%s: Quote '%d' has too many '%s' points, at most %d, saw %d (all ex tax)",
                __METHOD__,
                $quote->getId(),
                $pointsMax,
                $type,
                $quote->getPointsPoints()
            ));
        }

        $customerPoints = $provider->getUpdatedCustomerPointsBalance($store, $customer);
        $customerPointsIncludesTax = $provider->getCustomerPointsBalanceIncludesTax($store, $customer);

        $requestedPoints = $customerPointsIncludesTax ?
            $quote->getPointsPoints() + $quote->getPointsTaxPoints() :
            $quote->getPointsPoints();

        if($customerPoints < $requestedPoints) {
            throw new Exception(sprintf(
                "%s: Quote '%d' requests %d points %s, customer %d has %d available",
                __METHOD__,
                $quote->getId(),
                $requestedPoints,
                $customerPointsIncludesTax ? "including tax" : "excluding tax",
                $customer->getId(),
                $customerPoints
            ));
        }

        $provider->onQuoteSubmitBefore($order, $quote);
    }

    /**
     * @param Order|Quote $obj
     */
    protected function getQuoteProvider($obj): ProviderInterface {
        $type = $obj->getPointsType();
        $store = $obj->getStore();
        $helper = Mage::helper("points_core");

        if( ! $type) {
            throw new Exception(sprintf(
                "%s: Amount with no type on %s %d",
                __METHOD__,
                get_class($obj),
                $obj->getId()
            ));
        }

        $provider = $helper->getTypeProvider($store, $type);

        if( ! $provider) {
            throw new Exception(sprintf(
                "%s: Provider for points type '%s' not found for %s %d",
                __METHOD__,
                $type,
                get_class($obj),
                $obj->getId()
            ));
        }

        return $provider;
    }

    public function salesOrderPlaceBefore(Varien_Event_Observer $observer): void {
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($order->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($order);

        $provider->onOrderPlaceBefore($order);
    }

    public function salesOrderPlaceAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($order->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($order);

        $provider->onOrderPlaceEnd($order);
    }

    public function salesQuoteSubmitFailure(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($quote->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($quote);

        $provider->onQuoteSubmitFailure($order, $quote);
    }

    public function salesQuoteSubmitAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Quote
         */
        $quote = $observer->getQuote();
        /**
         * @var Order
         */
        $order = $observer->getOrder();

        if(abs($order->getBasePointsAmount()) < 0.01) {
            return;
        }

        $provider = $this->getQuoteProvider($order);

        $provider->onQuoteSubmitAfter($order, $quote);
    }

    /**
     * Observer listening to `sales_order_invoice_save_after` to update deposit amount invoiced in order.
     */
    public function invoiceSaveAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Invoice
         */
        $invoice = $observer->getInvoice();
        /**
         * @var Order
         */
        $order = $invoice->getOrder();

        if($invoice->getBasePointsAmount()) {
            $order->setPointsPointsInvoiced($order->getPointsPointsInvoiced() + $invoice->getPointsPoints());
            $order->setPointsTaxPointsInvoiced($order->getPointsTaxPointsInvoiced() + $invoice->getPointsTaxPoints());
            $order->setPointsAmountInvoiced($order->getPointsAmountInvoiced() + $invoice->getPointsAmount());
            $order->setBasePointsAmountInvoiced($order->getBasePointsAmountInvoiced() + $invoice->getBasePointsAmount());
            $order->setPointsTaxAmountInvoiced($order->getPointsTaxAmountInvoiced() + $invoice->getPointsTaxAmount());
            $order->setBasePointsTaxAmountInvoiced($order->getBasePointsTaxAmountInvoiced() + $invoice->getBasePointsTaxAmount());
        }
    }

    /**
     * Observer listening to `sales_order_creditmemo_save_after` to update deposit amount refunded in order.
     */
    public function creditmemoSaveAfter(Varien_Event_Observer $observer): void {
        /**
         * @var Creditmemo
         */
        $creditmemo = $observer->getCreditmemo();
        /**
         * @var Order
         */
        $order = $creditmemo->getOrder();

        if($creditmemo->getBasePointsAmount()) {
            $order->setPointsPointsRefunded($order->getPointsPointsRefunded() + $creditmemo->getPointsPoints());
            $order->setPointsTaxPointsRefunded($order->getPointsTaxPointsRefunded() + $creditmemo->getPointsTaxPoints());
            $order->setPointsAmountRefunded($order->getPointsAmountRefunded() + $creditmemo->getPointsAmount());
            $order->setBasePointsAmountRefunded($order->getBasePointsAmountRefunded() + $creditmemo->getBasePointsAmount());
            $order->setPointsTaxAmountRefunded($order->getPointsTaxAmountRefunded() + $creditmemo->getPointsTaxAmount());
            $order->setBasePointsTaxAmountRefunded($order->getBasePointsTaxAmountRefunded() + $creditmemo->getBasePointsTaxAmount());
        }
    }
}
