<?php

declare(strict_types=1);

namespace Points\Core;

/**
 * @template T as numeric
 */
class Amount {
    /**
     * @var T
     */
    protected $value;

    /**
     * @var T
     */
    protected $tax;

    /**
     * If set to true, $value includes tax and total is $value, otherwise
     * $value + $tax.
     *
     * @var bool
     */
    protected $valueIncludesTax;

    /**
     * @template U as numeric
     * @param Amount<U> $amount
     * @return U
     */
    public static function tax(Amount $amount) {
        return $amount->getTax();
    }

    /**
     * @template U as numeric
     * @param Amount<U> $amount
     * @return U
     */
    public static function value(Amount $amount) {
        return $amount->getValue();
    }

    /**
     * @template U as numeric
     * @param Amount<U> $amount
     * @return U
     */
    public static function totalInclTax(Amount $amount) {
        return $amount->getTotalInclTax();
    }

    /**
     * @template U as numeric
     * @param Amount<U> $amount
     * @return U
     */
    public static function totalExclTax(Amount $amount) {
        return $amount->getTotalExclTax();
    }

    /**
     * @param T $value
     * @param T $tax
     */
    public function __construct($value, bool $valueIncludesTax, $tax) {
        $this->value = $value;
        $this->valueIncludesTax = $valueIncludesTax;
        $this->tax = $tax;
    }

    /**
     * @return T
     */
    public function getValue() {
        return $this->value;
    }

    /**
     * @return T
     */
    public function getTax() {
        return $this->tax;
    }

    public function getValueIncludesTax(): bool {
        return $this->valueIncludesTax;
    }

    /**
     * @return T
     */
    public function getTotalInclTax() {
        /**
         * @var T
         */
        return $this->valueIncludesTax ? $this->value : $this->value + $this->tax;
    }

    /**
     * @return T
     */
    public function getTotalExclTax() {
        /**
         * @var T
         */
        return $this->valueIncludesTax ? $this->value - $this->tax : $this->value;
    }
}
