<?php

declare(strict_types=1);

namespace Points\Core;

use function Points\Core\calculateRoundingSpread;

use Mage;

use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;

class AmountsTest extends TestCase {
    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::init();
    }

    public function testEmpty(): void {
        $calculated = calculateRoundingSpread([]);

        $this->assertEquals([], $calculated);
    }

    public function testSingles(): void {
        $this->assertEquals([ "val" => 1 ], calculateRoundingSpread([ "val" => 0.9]));
        $this->assertEquals([ "val" => 1 ], calculateRoundingSpread([ "val" => 0.1]));
        $this->assertEquals([ "val" => 1 ], calculateRoundingSpread([ "val" => 0.4]));
        $this->assertEquals([ "val" => 1 ], calculateRoundingSpread([ "val" => 0.5]));
    }

    /*
    public function testOrder(): void {
        $initial = [
            "idx_0" => 266.4,
            "idx_1" => 399.8,
        ];

        $calculated = calculateRoundingSpread($initial);

        $this->assertSame([
            "idx_0" => 267,
            "idx_1" => 400,
        ], $calculated);
    }
     */

    public function testOne(): void {
        $total = 999;
        $calculated = calculateRoundingSpread([
            "amount" => 799.2,
            "tax" => 199.8,
        ]);

        $this->assertEquals([
            "amount" => 799,
            "tax" => 200,
        ], $calculated);
        $this->assertEquals($total, array_sum($calculated));
    }

    public function testNines(): void {
        $calculated = calculateRoundingSpread([
            "idx_0" => 3330.0,
            "idx_1" => 7494.999999999999,
        ]);

        $this->assertSame([
            "idx_0" => 3330,
            "idx_1" => 7495,
        ], $calculated);
    }

    public function testMultiple(): void {
        $values = [
            "one" => 5.8,
            "two" => 1.6,
            "three" => 5.6,
        ];
        $total = (int)ceil(array_sum($values));
        $calculated = calculateRoundingSpread($values);

        $this->assertEquals([
            "one" => 6,
            "two" => 2,
            "three" => 5,
        ], $calculated);
        $this->assertEquals($total, array_sum($calculated));
    }

    public function testSpreadAmount(): void {
        $this->assertEquals(spreadAmount([
            new Amount(1000, true, 0),
            new Amount(1332, true, 266.4),
            new Amount(100, true, 19.97008),
        ]), [
            new Amount(1000, true, 0),
            new Amount(1332, true, 267),
            new Amount(100, true, 20),
        ]);
    }

    /**
     * @psalm-suppress InvalidArgument Psalm defaults to constants instead of int so ->add expects Amount<0>.
     */
    public function testAdd(): void {
        $this->assertEquals(
            (new Amount(0, true, 0))->add(new Amount(5, true, 0)),
            new Amount(5, true, 0)
        );
        $this->assertEquals(
            (new Amount(0, true, 0))->add(new Amount(5, true, 4)),
            new Amount(5, true, 4)
        );
        $this->assertEquals(
            (new Amount(0, true, 0))->add(new Amount(5, false, 4)),
            new Amount(9, true, 4)
        );
        $this->assertEquals(
            (new Amount(0, false, 0))->add(new Amount(5, true, 4)),
            new Amount(1, false, 4)
        );
    }

    public function testMultiply(): void {
        $this->assertEquals(
            (new Amount(5, true, 4))->multiply(0, 0),
            new Amount(0, true, 0)
        );
        $this->assertEquals(
            (new Amount(0, true, 0))->multiply(2, 0),
            new Amount(0, true, 0)
        );
        $this->assertEquals(
            (new Amount(5, true, 4))->multiply(2, 0),
            new Amount(10, true, 8)
        );
        $this->assertEquals(
            (new Amount(5, false, 4))->multiply(2, 0),
            new Amount(10, false, 8)
        );
        $this->assertEquals(
            (new Amount(5, true, 4))->multiply(0.5, 1),
            new Amount(2.5, true, 2)
        );
        $this->assertEquals(
            (new Amount(5, false, 4))->multiply(0.5, 1),
            new Amount(2.5, false, 2)
        );
        $this->assertEquals(
            (new Amount(5, false, 4))->multiply(0.5, 0),
            new Amount(3, false, 2)
        );
    }

    public function testSpreadPriceBadRouding(): void {
        $this->assertEquals([
            189,
            782,
            28,
        ], calculateRoundingSpread([
            188.561721,
            782.125708,
            28.312571,
        ]));
    }
}
