<?php

declare(strict_types=1);

namespace Points\Core;

use Mage_Tax_Model_Config;
use Mage_Sales_Model_Quote;
use Mage_Sales_Model_Quote_Item;
use Mage_Sales_Model_Quote_Address;
use Mage_Sales_Model_Quote_Address_Item;
use Exception;
use Mage;

/**
 * Function wrapping time, which can be overriden in tests.
 */
function getTime(): int {
    return time();
}

function truncate3(float $a): float {
    return ((int)($a * 1000)) / 1000;
}

/**
 * Spreads the rounding error across multiple values, preserving item order.
 * @param Array<float> $amounts
 * @return Array<int>
 */
function calculateRoundingSpread(array $amounts): array {
    // We need to store the key order to reconstruct later
    $keyOrder = [];

    foreach(array_keys($amounts) as $i => $k) {
        $keyOrder[$k] = "_".$i;
    }

    $amounts = array_combine($keyOrder, $amounts);

    // Truncate to 3 decimal digits to avoid tiny float errors,
    // could possibly be solved by using a diff vs ceil/floor with 0.01
    $amounts = array_map("Points\\Core\\truncate3", $amounts);

    $total = (int)ceil(array_sum($amounts));
    $totalLow = array_sum($amounts);
    $totalErr = $total - $totalLow;

    // Sort them with the lowest rouding error vs floor first
    uasort($amounts, function(float $a, float $b) {
        $errA = $a - floor($a);
        $errB = $b - floor($b);
        $diff = $errA - $errB;

        return $diff > 0 ? 1 : ($diff < 0 ? -1 : 0);
    });

    // If we have error to correct we want to start to round upwards on
    // the largest error vs floor first
    if($totalErr > 0) {
        $amounts = array_reverse($amounts);
    }

    // Walk over the existing values and round them up as long as
    // we have remaining error
    $amountsRounded = [];

    foreach($amounts as $k => $a) {
        // Correct error upwards if we still have some of the total error to
        // spend, take the float error into account
        if($totalErr > 0) {
            $rounded = ceil($a);
            $totalErr -= $rounded - $a;
        }
        else {
            // We have to increase the error since we are flooring
            $rounded = floor($a);
            $totalErr -= $rounded - $a;
        }

        $amountsRounded[$k] = (int)$rounded;
    }

    if(abs($totalErr) >= 0.01) {
        $msg = sprintf(
            "%s: Failed to properly spread rounding error, totalErr: %f, on: %s",
            __FUNCTION__,
            $totalErr,
            json_encode($amounts)
        );

        Mage::log($msg);

        if(Mage::getIsDeveloperMode()) {
            throw new Exception($msg);
        }
    }

    $originalOrderRounded = [];

    foreach($keyOrder as $k => $idx) {
        $originalOrderRounded[$k] = $amountsRounded[$idx];
    }

    return $originalOrderRounded;
}

/**
 * Spreads the rounding into decimal numbers with specified precision.
 *
 * @template T as numeric
 * @param Array<T> $prices
 * @return Array<float>
 */
function spreadPrice(array $prices, int $precision = 2): array {
    $multiplier = pow(10, $precision);

    return array_map(function(int $a) use($multiplier): float {
        return $a / $multiplier;
    }, calculateRoundingSpread(array_map(
        /**
         * @param numeric $a
         * @return float
         */
        function($a) use($multiplier) {
            return $a * $multiplier;
        },
        $prices
    )));
}

/**
 * @template P as int
 * @param Array<Amount<float|int>> $items
 * @param P $precision
 * @return Array<Amount<float>>
 * @psalm-return Array<Amount<(P is 0 ? int : float)>>
 */
function spreadAmount(array $items, int $precision = 0): array {
    $keys = array_keys($items);
    $amounts = spreadPrice(array_map(function($a) { return $a->getValue(); }, $items), $precision);
    $taxes = spreadPrice(array_map(function($a) { return $a->getTax(); }, $items), $precision);

    return array_combine(
        // Preserve keys
        $keys,
        array_map(
            /**
             * @param array-key $k
             */
            function($k) use($amounts, $taxes, $items, $precision): Amount {
                return new Amount(
                    $precision === 0 ? (int)$amounts[$k] : $amounts[$k],
                    $items[$k]->getValueIncludesTax(),
                    $precision === 0 ? (int)$taxes[$k] : $taxes[$k]
                );
            },
            $keys
        )
    );
}

/**
 * Returns the smallest amount compared based on $incVat.
 *
 * NOTE: It is recommended to use Amount::scaledBy to obtain amounts with
 * proportional tax-rates from a larger total sum.
 *
 * @template T of int|float
 * @param non-empty-list<Amount<T>> $amounts
 * @return Amount<T>
 */
function amountMin(array $amounts, bool $incVat): Amount {
    $min = array_pop($amounts);

    foreach($amounts as $amount) {
        if($min->getTotalAndTax($incVat) > $amount->getTotalAndTax($incVat)) {
            $min = $amount;
        }
    }

    return $min;
}

/**
 * Returns the largest amount compared based on $incVat.
 *
 * NOTE: It is recommended to use Amount::scaledBy to obtain amounts with
 * proportional tax-rates from a larger total sum.
 *
 * @template T of int|float
 * @param non-empty-list<Amount<T>> $amounts
 * @return Amount<T>
 */
function amountMax(array $amounts, bool $incVat): Amount {
    $max = array_pop($amounts);

    foreach($amounts as $amount) {
        if($max->getTotalAndTax($incVat) < $amount->getTotalAndTax($incVat)) {
            $max = $amount;
        }
    }

    return $max;
}

/**
 * @param Mage_Sales_Model_Quote $quote
 * @return bool
 */
function quotePriceIncludesTax($quote) {
    $store = $quote->getStore();

    return (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
}

/**
 * @param Mage_Sales_Model_Quote_Item|Mage_Sales_Model_Quote_Address_Item $item
 * @return bool
 */
function quoteItemPriceIncludesTax($item) {
    $quote = $item->getQuote();

    assert($quote !== null);

    return quotePriceIncludesTax($quote);
}
