<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use Mage;
use Mage_Catalog_Model_Product;
use Mage_Core_Model_Store;
use Mage_Tax_Model_Config;
use MageQL\Context;

class ProductPointsTotal implements TotalInterface {
    /**
     * @var Mage_Catalog_Model_Product
     */
    protected $product;
    /**
     * @var int
     */
    protected $price;

    public function __construct(
        Mage_Catalog_Model_Product $product,
        int $price
    ) {
        $this->product = $product;
        $this->price = $price;
    }

    protected function getTaxConversionRate(
        Mage_Core_Model_Store $store,
        bool $pricesIncludeTax
    ): float {
        /**
         * @var ?float
         */
        $percent = $this->product->getTaxPercent();
        $taxClassId = $this->product->getTaxClassId();

        if (is_null($percent)) {
            if ($taxClassId) {
                $calc = Mage::getSingleton("tax/calculation");
                $request = $calc->getRateRequest(null, null, null, $store);

                $request->setProductClassId($taxClassId);

                $percent = $calc->getRate($request);
            }
        }

        $taxRate = ($percent ?: 0) / 100;

        return $pricesIncludeTax ? $taxRate / (1 + $taxRate) : $taxRate;
    }

    public function getIncVat(
        Context $ctx
    ): int {
        $store = $ctx->getStore();
        $pricesIncludeTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);

        return $pricesIncludeTax
            ? $this->price
            : (int)ceil($this->price + $this->price * $this->getTaxConversionRate($store, $pricesIncludeTax));
    }

    public function getExVat(
        Context $ctx
    ): int {
        $store = $ctx->getStore();
        $pricesIncludeTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);

        return $pricesIncludeTax
            ? (int)ceil($this->price - $this->price * $this->getTaxConversionRate($store, $pricesIncludeTax))
            : $this->price;
    }

    public function getVat(
        Context $ctx
    ): int {
        $store = $ctx->getStore();
        $pricesIncludeTax = (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);

        return (int)ceil($this->price * $this->getTaxConversionRate($store, $pricesIncludeTax));
    }
}

