<?php

declare(strict_types=1);

namespace Points\Core;

use Mage_Core_Model_Store;
use Mage_Customer_Model_Customer;
use Mage_Catalog_Model_Product;
use Mage_Sales_Model_Quote;
use Mage_Sales_Model_Order;
use Mage_Sales_Model_Quote_Address;

/**
 * Handled as a singleton, make sure to use the store provided by the Quote or
 * Order instance if no store-instance is supplied.
 */
interface ProviderInterface {
    /**
     * Returns true if this point provider is enabled for this store.
     *
     * TODO: Maybe also provide customer?
     */
    public function isEnabled(Mage_Core_Model_Store $store): bool;

    /**
     * Returns true if this point payment provider can be used with the supplied quote.
     */
    public function appliesTo(Mage_Sales_Model_Quote $quote): bool;

    /**
     * Returns the label show to the customer for this point type.
     */
    public function getLabel(Mage_Core_Model_Store $store): string;

    /**
     * Returns the amount of points the shipping would cost for the given address.
     */
    public function getQuoteShippingPrice(Mage_Sales_Model_Quote_Address $address): ?int;

    /**
     * Returns the fallback currency-to-point conversion-multiplier for products, used by product import.
     */
    public function getProductFallbackRate(
        Mage_Core_Model_Store $store,
        Mage_Catalog_Model_Product $product
    ): ?float;

    /**
     * Returns the approximate amount of points available to the customer to be
     * used for the display of available points, null indicates the customer is
     * not eligible for this points currency.
     *
     * NOTE: Does not need to validate the actual amount is present, this
     * method is just to indicate the available value to the customer.
     */
    public function getCustomerPointsBalance(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): int;

    /**
     * Returns true if the customer points balance is used to pay for taxes,
     * if included tax will count towards the total points to deduct.
     *
     * NOTE: Default should be true.
     */
    public function getCustomerPointsBalanceIncludesTax(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): bool;

    /**
     * Return true if the supplied customer is allowed to redeem points of this
     * type in the supplied store.
     */
    public function getCustomerRedemptionAllowed(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): bool;

    /**
     * Updates the customer point balance from any remote points source and
     * returns the new balance.
     */
    public function getUpdatedCustomerPointsBalance(
        Mage_Core_Model_Store $store,
        Mage_Customer_Model_Customer $customer
    ): int;

    /**
     * Perform extra validation or preparation before the actual quote payment
     * is initiated.
     */
    public function onQuoteSubmitBefore(
        Mage_Sales_Model_Order $order,
        Mage_Sales_Model_Quote $quote
    ): void;

    /**
     * Perform payment or authorization of points, crashing here will abort the
     * order and cancel the already pending payment.
     *
     * This is triggered BEFORE payment is performed.
     *
     * NOTE: This is usually run inside of a transaction when placing an order,
     * all failures will rollback any database-writes to the `core_write` adapter.
     */
    public function onOrderPlaceBefore(Mage_Sales_Model_Order $order): void;

    /**
     * Perform payment of points, crashing here will abort the order and cancel
     * the already pending payment.
     *
     * This is triggered AFTER payment is performed.
     *
     * NOTE: This is usually run inside of a transaction when placing an order,
     * all failures will rollback any database-writes to the `core_write` adapter.
     */
    public function onOrderPlaceEnd(Mage_Sales_Model_Order $order): void;

    /**
     * Rollback anything previously done, payment or similar failed.
     *
     * Database writes should already have been rolled back, but make sure.
     */
    public function onQuoteSubmitFailure(
        Mage_Sales_Model_Order $order,
        Mage_Sales_Model_Quote $quote
    ): void;

    /**
     * This is always run if we did not previously error out.
     */
    public function onQuoteSubmitAfter(
        Mage_Sales_Model_Order $order,
        Mage_Sales_Model_Quote $quote
    ): void;
}
