<?php

declare(strict_types=1);

use function Points\Core\calculateRoundingSpread;
use function Points\Core\spreadPrice;
use function Points\Core\quotePriceIncludesTax;
use function Points\Core\amountMin;
use function Points\Core\amountMax;

use Points\Core\Total\Calculator;
use Points\Core\Total\QuoteAddress as QuoteAddressTotal;
use Points\Core\Total\Item as ItemTotal;
use Points\Core\Total\Shipping as ShippingTotal;
use Points\Core\Total\SelectedPoints;

use Points\Core\Amount;
use Points\Core\ProviderInterface;

/**
 * @psalm-type Limit array{min:int, max:int}
 * @psalm-type Rate array{code:string, title:string, percent:float, position:string|int, priority:string|int, rule_id:string|int}
 * @psalm-type ItemAppliedRate array{id:string, percent:float, rates:list<Rate>}
 * @psalm-type AddressAppliedRate array{rates:list<Rate>, percent:float, id:string, process:int, amount:float, base_amount:float}
 */
class Points_Core_Model_Sales_Total_Quote_Points extends Mage_Sales_Model_Quote_Address_Total_Abstract
{
    public const LOG_CHANNEL = "points_core_calculation";
    public const PRECISION = 6;

    protected $_code = "points";

    public function collect(Mage_Sales_Model_Quote_Address $address) {
        parent::collect($address);

        $this->resetAddress($address);

        $items = $this->_getAddressItems($address);

        if(count($items) === 0) {
            // It is already reset in observer
            return $this;
        }

        $helper = Mage::helper("points_core");
        $quote = $address->getQuote();
        $type = $quote->getPointsType();

        if( ! $type) {
            return $this;
        }

        $store = $quote->getStore();
        $customer = $quote->getCustomer();
        $provider = $helper->getTypeProvider($type);

        if( ! $provider ||
            ! $provider->appliesTo($quote) ||
            ! $customer->getId() ||
            ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            return $this;
        }

        $calculator = new Calculator();
        $quotePriceIncludesTax = quotePriceIncludesTax($quote);
        $incVat = $quote->getPointsWantedIncludesTax() ?? $quotePriceIncludesTax;
        $addressTotal = $calculator->fromQuoteAddress($address, $type, $provider);
        $requestedPoints = $this->getRequestedPoints($addressTotal, $provider, $type, $incVat);
        $requestedTotal = $calculator->calculateSelected($addressTotal, $requestedPoints, $incVat);
        $pointsSelected = $requestedTotal->getPoints()->getSelected();

        assert($requestedPoints === $pointsSelected->getTotalAndTax($incVat), sprintf(
            "PointsCore: Assert 'requestedPoints (%d) === pointsSelected->getTotalAndTax(%s) (%d)' failed for quote (%d) address (%d)",
            $requestedPoints,
            $incVat ? "true" : "false",
            $pointsSelected->getTotalAndTax($incVat),
            (int)$quote->getId(),
            (int)$address->getId()
        ));

        $this->collectItems($requestedTotal);
        $this->collectDiscount($requestedTotal, $incVat);

        $excluded = $requestedTotal->getExcluded()->getTotal();
        $included = $requestedTotal->getIncluded()->getTotal();
        $pointsValue = $requestedTotal->getPoints()->getValue();

        $precision = self::PRECISION;

        // addressBaseTotal includes tax
        $addressBaseTotal = round(array_sum($address->getAllBaseTotalAmounts()), $precision);

        /** @var Amount<float> */
        $baseAmount = new Amount(0.0, $incVat, 0.0);

        foreach($requestedTotal->getTotals() as $t) {
            $pointTotal = $t->getPointTotalAmount();

            if($pointTotal) {
                $baseAmount = $baseAmount->add($pointTotal);
            }
        }


        // Sanity checks

        assert(abs($included->getTotalInclTax() + $excluded->getTotalInclTax() - $addressBaseTotal) < 0.01, sprintf(
            "PointsCore: Assert 'includedTotal (%.{$precision}f) + excludedTotal (%.{$precision}f) - addressTotal (%.{$precision}f)' failed for quote (%d) address (%d)",
            $included->getTotalInclTax(),
            $excluded->getTotalInclTax(),
            $addressBaseTotal,
            (int)$quote->getId(),
            (int)$address->getId()
        ));

        $baseAmountTax = round($baseAmount->getTax(), $precision);
        $baseTotalAmountTax = round($address->getBaseTotalAmount("tax"), $precision);
        assert($baseAmountTax <= $baseTotalAmountTax, sprintf(
            "PointsCore: Assert 'baseAmount->getTax (%.{$precision}f) <= address->getBaseTotalAmount(tax) (%.{$precision}f)' failed for quote (%d) address (%d)",
            $baseAmountTax,
            $baseTotalAmountTax,
            (int)$quote->getId(),
            (int)$address->getId()
        ));

        $baseAmountTotalInclTax = round($baseAmount->getTotalInclTax(), $precision);
        assert($baseAmountTotalInclTax <= $addressBaseTotal, sprintf(
            "PointsCore: Assert 'baseAmount->getTotalInclTax (%.{$precision}f) <= addressBaseTotal (%.{$precision}f)' failed for quote (%d) address (%d)",
            $baseAmountTotalInclTax,
            $addressBaseTotal,
            (int)$quote->getId(),
            (int)$address->getId()
        ));

        $amount = $store->convertPrice($baseAmount->getTotalExclTax(), false);
        $taxAmount = $store->convertPrice($baseAmount->getTax(), false);

        $address->setPointsPoints($pointsSelected->getTotalExclTax());
        $address->setPointsTaxPoints($pointsSelected->getTax());
        $address->setTotalAmount($this->getCode(), -$amount);
        $address->setBaseTotalAmount($this->getCode(), -$baseAmount->getTotalExclTax());

        // Reduce tax with what we paid
        $address->addTotalAmount("tax", -$taxAmount);
        $address->addBaseTotalAmount("tax", -$baseAmount->getTax());
        $address->setPointsTaxAmount($taxAmount);
        $address->setBasePointsTaxAmount($baseAmount->getTax());

        $address->setPointsValue((int)$pointsValue->getTotalExclTax());
        $address->setPointsTaxValue((int)$pointsValue->getTax());
        $address->setPointsAmountIncluded($included->getTotalExclTax());
        $address->setPointsTaxAmountIncluded($included->getTax());
        $address->setPointsAmountExcluded($excluded->getTotalExclTax());
        $address->setPointsTaxAmountExcluded($excluded->getTax());

        $quote->setPointsWanted((int)$quote->getPointsWanted() - $requestedPoints);

        return $this;
    }

    public function fetch(Mage_Sales_Model_Quote_Address $address) {
        $address->addTotal([
            "code"           => $this->getCode(),
            "value"          => $address->getPointsAmount(),
            "value_incl_tax" => $address->getPointsAmount() + $address->getPointsTaxAmount(),
            "value_excl_tax" => $address->getPointsAmount(),
        ]);

        return [];
    }

    private function resetAddress(
        Mage_Sales_Model_Quote_Address $address
    ): void {
        $address->setPointsValue(0);
        $address->setPointsTaxValue(0);
        $address->setPointsPoints(0);
        $address->setPointsTaxPoints(0);
        $address->setPointsAmount(0.0);
        $address->setPointsTaxAmount(0.0);
        $address->setBasePointsAmount(0.0);
        $address->setBasePointsTaxAmount(0.0);

        $address->setPointsAmountIncluded(0);
        $address->setPointsTaxAmountIncluded(0);
        $address->setPointsAmountExcluded(0);
        $address->setPointsTaxAmountExcluded(0);

        $address->setPointsShippingValue(null);
        $address->setPointsShippingTaxValue(null);
        $address->setPointsShippingPoints(0);
        $address->setPointsShippingTaxPoints(0);
        $address->setPointsShippingTotal(0.0);
        $address->setPointsShippingTaxTotal(0.0);
        $address->setBasePointsShippingTotal(0.0);
        $address->setBasePointsShippingTaxTotal(0.0);

        $address->setPointsDiscountValue(0);
        $address->setPointsDiscountTaxValue(0);
        $address->setPointsDiscountPoints(0);
        $address->setPointsDiscountTaxPoints(0);
        $address->setPointsDiscountTotal(0.0);
        $address->setPointsDiscountTaxTotal(0.0);
        $address->setBasePointsDiscountTotal(0.0);
        $address->setBasePointsDiscountTaxTotal(0.0);

        foreach($this->_getAddressItems($address) as $i) {
            // Reset to make sure we do not populate if it does not have a point price
            $i->setPointsRowValue(null);
            $i->setPointsRowTaxValue(null);
            $i->setPointsRowPoints(0);
            $i->setPointsRowTaxPoints(0);
            $i->setPointsRowTotal(0.0);
            $i->setPointsRowTaxTotal(0.0);
            $i->setBasePointsRowTotal(0.0);
            $i->setBasePointsRowTaxTotal(0.0);
        }
    }

    private function getRequestedPoints(
        QuoteAddressTotal $totals,
        ProviderInterface $provider,
        string $type,
        bool $incVat
    ): int {
        $quote = $totals->getAddress()->getQuote();
        $store = $quote->getStore();
        $customer = $quote->getCustomer();

        // TODO: Break out somehow
        $customerPoints = $provider->getCustomerPointsBalance($store, $customer);
        $customerPointsIncludesTax = $provider->getCustomerPointsBalanceIncludesTax($store, $customer);

        $points = $totals->getPoints();
        $pointsValue = $points->getValue();
        $pointsMax = $points->getMax();
        $pointsMin = $points->getMin();

        $totalLimit = Mage::getResourceModel("points_core/limit_total")->getMatchingLimit($store, $customer->getGroupId(), $type);
        $orderLimit = Mage::getResourceModel("points_core/limit_order")->getMatchingLimit($store, $customer->getGroupId(), $type);

        $maximum = [
            $pointsMax,
            $pointsValue->scaleTo($customerPoints, $customerPointsIncludesTax, 0),
            $pointsValue->scaleTo((int)$quote->getPointsWanted(), $incVat, 0),
        ];
        $minimum = [
            $pointsMin,
        ];

        if($totalLimit) {
            $totalLimitRemaining = $totalLimit->getRemaining($customer);

            if($totalLimitRemaining !== null) {
                $maximum[] = $pointsValue->scaleTo($totalLimitRemaining, (bool)$totalLimit->getIncludesTax(), 0);

            }
        }

        if($orderLimit) {
            $orderIncVat = (bool)$orderLimit->getIncludesTax();
            $valueTotal = $pointsValue->getTotalAndTax($orderIncVat);
            $orderMax = $orderLimit->getTotalMax($valueTotal);
            $orderMin = $orderLimit->getTotalMin($valueTotal);

            $maximum[] = $pointsValue->scaleTo($orderMax, $orderIncVat, 0);
            $minimum[] = $pointsValue->scaleTo($orderMin, $orderIncVat, 0);
        }

        $requestedPoints = amountMin($maximum, $incVat);
        $minimumPoints = amountMax($minimum, $incVat);

        if($requestedPoints->getTotalAndTax($incVat) >= $minimumPoints->getTotalAndTax($incVat)) {
            return $requestedPoints->getTotalAndTax($incVat);
        }

        // Zero it since we can't actually use any
        return 0;
    }

    /**
     * Collects itemized totals for rows and shipping.
     */
    private function collectItems(
        QuoteAddressTotal $addressTotal
    ): void {
        $address = $addressTotal->getAddress();
        $quote = $address->getQuote();
        $store = $quote->getStore();

        foreach($addressTotal->getTotals() as $total) {
            $points = $total->getPoints();

            if( ! $points) {
                continue;
            }

            assert($points instanceof SelectedPoints);

            $rowValue = $points->getValue();
            $selected = $points->getSelected();
            $rowTotal = $total->getPointTotalAmount();

            assert($rowTotal !== null);

            $tax = $rowTotal->getTax();
            $baseTax = $store->convertPrice($tax, false);

            if($total instanceof ItemTotal) {
                $item = $total->getItem();

                $item->setPointsRowValue($rowValue->getTotalExclTax());
                $item->setPointsRowTaxValue($rowValue->getTax());
                $item->setPointsRowPoints($selected->getTotalExclTax());
                $item->setPointsRowTaxPoints($selected->getTax());
                $item->setPointsRowTotal($rowTotal->getTotalExclTax());
                $item->setPointsRowTaxTotal($tax);
                $item->setBasePointsRowTotal($store->convertPrice($rowTotal->getTotalExclTax(), false));
                $item->setBasePointsRowTaxTotal($baseTax);

                // We have to reduce the amount but also take the discount into account
                /** @var list<ItemAppliedRate> */
                $appliedRates = $item->getAppliedRates() ?? [];

                $this->saveAppliedTaxes($address, $appliedRates, -$baseTax, -$tax);
            }

            if($total instanceof ShippingTotal) {
                $address->setPointsShippingValue($rowValue->getTotalExclTax());
                $address->setPointsShippingTaxValue($rowValue->getTax());
                $address->setPointsShippingPoints($selected->getTotalExclTax());
                $address->setPointsShippingTaxPoints($selected->getTax());
                $address->setPointsShippingTotal($rowTotal->getTotalExclTax());
                $address->setPointsShippingTaxTotal($tax);
                $address->setBasePointsShippingTotal($store->convertPrice($rowTotal->getTotalExclTax(), false));
                $address->setBasePointsShippingTaxTotal($baseTax);

                // We have to recreate the tax calculation to obtain the used tax rate for shipping
                $custTaxClassId = (int)$address->getQuote()->getCustomerTaxClassId();
                $shippingTaxClass = (int)Mage::getStoreConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_TAX_CLASS, $store);
                $taxCalculationModel = Mage::getSingleton("tax/calculation");
                /** @var Mage_Sales_Model_Quote_Address|false|null $billingAddress */
                $billingAddress = $address->getQuote()->getBillingAddress();
                $request = $taxCalculationModel->getRateRequest(
                    $address,
                    $billingAddress,
                    $custTaxClassId,
                    $store
                );
                $request->setProductClassId($shippingTaxClass);

                /** @var list<ItemAppliedRate> */
                $appliedRates = $taxCalculationModel->getAppliedRates($request);

                $this->saveAppliedTaxes($address, $appliedRates, -$baseTax, -$tax);
            }
        }
    }

    /**
     * Collects discount total in points.
     */
    private function collectDiscount(
        QuoteAddressTotal $totals,
        bool $incVat
    ): void {
        $address = $totals->getAddress();
        $store = $address->getQuote()->getStore();

        $pointsSelected = $totals->getPoints()->getSelected();
        $pointsTotalValue = $totals->getPoints()->getValue()->getTotalAndTax($incVat);
        $fracPoints = $pointsTotalValue > 0 ? $pointsSelected->getTotalAndTax($incVat) / $pointsTotalValue : 0;

        $pointsDiscountValue = $totals->getPoints()->getDiscount();
        $baseDiscount = $totals->getIncluded()->getDiscountTotal();

        $pointsDiscount = $pointsDiscountValue->multiply($fracPoints, 0);
        $baseTotalAmount = $baseDiscount->multiply($fracPoints, 2);

        $totalAmount = $store->convertPrice($baseTotalAmount->getTotalExclTax(), false);
        $totalTaxAmount = $store->convertPrice($baseTotalAmount->getTax(), false);

        $address->setPointsDiscountPoints($pointsDiscount->getTotalExclTax());
        $address->setPointsDiscountTaxPoints($pointsDiscount->getTax());
        $address->setPointsDiscountValue($pointsDiscountValue->getTotalExclTax());
        $address->setPointsDiscountTaxValue($pointsDiscountValue->getTax());
        $address->setPointsDiscountTotal($totalAmount);
        $address->setPointsDiscountTaxTotal($totalTaxAmount);
        $address->setBasePointsDiscountTotal($baseTotalAmount->getTotalExclTax());
        $address->setBasePointsDiscountTaxTotal($baseTotalAmount->getTax());
    }

    /**
     * @param list<ItemAppliedRate> $appliedRates
     */
    private function saveAppliedTaxes(
        Mage_Sales_Model_Quote_Address $address,
        array $appliedRates,
        float $baseAmount,
        float $amount
    ): void {
        /**
         * @var Array<string, AddressAppliedRate>
         */
        $previouslyAppliedTaxes = $address->getAppliedTaxes();

        foreach($appliedRates as $row) {
            if (!isset($previouslyAppliedTaxes[$row["id"]])) {
                Mage::log(sprintf(
                    "%s: Could not modify applied taxes for quote address '%d' on quote '%d' with rule '%s' by '%f'.",
                    __METHOD__,
                    $address->getId() ?: 0,
                    $address->getQuote()->getId() ?: 0,
                    $row["id"],
                    $baseAmount
                ), Zend_Log::WARN, self::LOG_CHANNEL);
                continue;
            }

            // We have to round here since we can have lost precision through serialization
            $newAmount = round($previouslyAppliedTaxes[$row["id"]]["amount"], 3) + $amount;
            $newBaseAmount = round($previouslyAppliedTaxes[$row["id"]]["base_amount"], 3) + $baseAmount;

            $previouslyAppliedTaxes[$row["id"]]["amount"] = $newAmount;
            $previouslyAppliedTaxes[$row["id"]]["base_amount"] = $newBaseAmount;
        }

        $address->setAppliedTaxes($previouslyAppliedTaxes);
    }
}
