#!/usr/bin/env bash

# TODO:
# * Deduplicate container boot/run logic

# Error on command errors, missing variables, and make pipes fail if any of the involved commands fail
set -euo pipefail
# Required to make *.sql glob return empty array instead of literal string
shopt -s nullglob

if [ -f .env ]; then
	source .env
fi

DEBUG=
# Missing in Bash
GID=$(id -g)
SCRIPT_NAME=$(basename "$0")

PROJECT_NAME=${PROJECT_NAME:-project-$(basename "$PWD")}
MAGE_RUN_CODE=${MAGE_RUN_CODE:-magento}
COMPOSER_IMAGE=${COMPOSER_IMAGE:-composer:2}
NGINX_PORT=${NGINX_PORT:-8080}
DATABASE_NAME=${DATABASE_NAME:-testing}
COMPOSER_HOME=${COMPOSER_HOME:-$HOME/.config/composer}
MAGENTO_FOLDER=${MAGENTO_FOLDER:-$PWD/vendor/awardit/magento-lts}

DB_CONTAINER=$PROJECT_NAME-mariadb
FPM_CONTAINER=$PROJECT_NAME-fpm
NGINX_CONTAINER=$PROJECT_NAME-nginx
NETWORK=$PROJECT_NAME

declare -A IMAGES=(
	[php-test]="$PROJECT_NAME-test"
	[php-coverage]="$PROJECT_NAME-coverage"
	[php-fpm]="$PROJECT_NAME-fpm"
)

DOCKER_PARAMS=(
# We need to set UID and GID to allow is_writable to work
	-u "$UID:$GID"
	--rm
	-it
	-v "$PWD:/app"
	-w /app
)

COMPOSER_PARAMS=(
	"${DOCKER_PARAMS[@]}"
	-v "$COMPOSER_HOME:/tmp"
	# .ssh, passwd, and group volume to be able to use git in composer
	-v "$HOME/.ssh:/home/$USER/.ssh"
	-v /etc/passwd:/etc/passwd:ro
	-v /etc/group:/etc/group:ro
	"$COMPOSER_IMAGE"
)

PSALM_PARAMS=(
	"${DOCKER_PARAMS[@]}"
# --init for an init wrapper to manage INT
	--init
	"${IMAGES[php-test]}"
	php vendor/bin/psalm --show-info=true
)

PHPUNIT_PARAMS=(
	"${DOCKER_PARAMS[@]}"
	--init
	--network="$NETWORK"
	-e DB_HOST="$DB_CONTAINER"
	-e DB_NAME="$DATABASE_NAME"
	--read-only
	"${IMAGES[php-test]}"
)

DATABASE_VOLUMES=()

for f in "$PWD"/test/*.sql; do
	DATABASE_VOLUMES+=(-v "$f:/docker-entrypoint-initdb.d/$(basename "$f"):ro")
done

# Quicker health check since it is a test
# health check must use IP to avoid socket since socket is ready before TCP/IP
DATABASE_PARAMS=(
	-e MYSQL_ALLOW_EMPTY_PASSWORD=yes
	-e MYSQL_DATABASE="$DATABASE_NAME"
	--tmpfs /var/lib/mysql:rw
	--tmpfs /tmp:rw
	--network="$NETWORK"
	--health-cmd='mysql -h 127.0.0.1 -e "show databases;"'
	--health-interval=2s
	"${DATABASE_VOLUMES[@]}" mariadb
)

# Use fcgi to check if it is alive, should return status once running if
# pm.status_path = /status is set properly
FPM_PARAMS=(
	"${DOCKER_PARAMS[@]}"
	--init
	--network="$NETWORK"
	--name "$FPM_CONTAINER"
	--health-interval=2s
	"${IMAGES[php-fpm]}"
)

NGINX_PARAMS=(
	--network="$NETWORK"
	-e PHP_FPM_HOST="$FPM_CONTAINER"
	-e MAGE_RUN_CODE="$MAGE_RUN_CODE"
	-v "$PWD:/app:ro"
	-v "$PWD/dev/nginx:/etc/nginx/templates:ro"
	-p "$NGINX_PORT:80"
	--health-cmd="curl -Iq http://localhost:80"
	--health-interval=5s
	nginx:alpine
)

# Echos parameters if debug is enabled
debug_echo() {
	if [ -n "$DEBUG" ]; then
		echo "$@"
	fi
}

debug_run_cmd() {
	local first=""

	if [ -n "$DEBUG" ]; then
		for var in "$@"; do
			printf "%s'%s'" "$first" "$var"
			first=" "
		done

		echo ""
	fi

	"$@"
}

create_image() {
	local image="${IMAGES[$1]}"
	local output=()

	if [ -n "$DEBUG" ]; then
		output=(--progress=plain)
	fi

	if [ -z "$image" ]; then
		echo "Unknown image $1"; exit 1;
	fi

	if docker image inspect "$image" >/dev/null 2>&1; then
		debug_echo "Reusing $1 image"
	else
		# Pipe Dockerfile to avoid context
		debug_run_cmd docker build "${output[@]}" --target "$1" -t "$image" - < ./Dockerfile
	fi
}

is_container_running() {
	[ "$( docker container inspect -f '{{.State.Running}}' "$1" 2>/dev/null )" == "true" ]
}

start_container() {
	debug_run_cmd docker run -d --rm --name "$@"

	while [ "$(docker inspect --format "{{.State.Health.Status }}" "$1")" != "healthy" ]; do
		printf "."
		sleep 1
	done

	echo ""
}

sub_help() {
	echo "Usage: $SCRIPT_NAME [-v] <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    composer   Run composer"
	echo "    test       Run tests"
	echo "    psalm      Run psalm"
	echo "    phpunit    Run phpunit"
	echo "    coverage   Run phpunit and record code coverage"
	echo "    syntax     Run PHP syntax check"
	echo "    network    Manage the test network"
	echo "    database   Manage the test database container"
	echo "    fpm        Manage the test PHP-FPM container"
	echo "    nginx      Manage the test Nginx container"
	echo "    start      Starts all containers"
	echo "    stop       Stops all containers, removes all networks"
	echo "               and clears the cache"
	echo "    redeploy   Redeploys all magento modules (copies/symlinks)"
	echo "    clean      Removes containers, images, networks, unused volumes"
	echo "    distclean  Removes containers, images, networks, unused volumes,"
	echo "               coverage, vendor, and magento folders, composer.lock"
	echo ""
	echo "Options:"
	echo "    -v --verbose  Activates verbose print of containers"
	echo ""
	echo "Environment Variables:"
	echo "    PROJECT_NAME   The basename used for containers and images"
	echo "        default: project-\$DIR"
	echo "    MAGE_RUN_CODE  The run code forwarded to Nginx/FPM"
	echo "        default: magento"
	echo "    COMPOSER_IMAGE Docker image for Composer"
	echo "        default: composer:2"
	echo "    NGINX_PORT     Local port mapped to Nginx container port 80"
	echo "        default: 8080"
	echo "    DATABASE_NAME  MariaDB database name to use"
	echo "        default: testing"
	echo ""
	echo "If a .env file exists in the current working directory this will be"
	echo "loaded and populate environment variables for the command."
	echo ""
	echo "For help with each subcommand run:"
	echo "$SCRIPT_NAME <subcommand> -h|--help"
	echo ""
}

sub_syntax() {
	case "${1:-}" in
		"-h" | "--help")
			echo "Usage: $SCRIPT_NAME [-a]"
			echo ""
			echo "Options:"
			echo "    -a --all  Runs PHP syntax check on all .php files in the"
			echo "              current folder including files outside source control"
			echo ""

			;;
		"-a" | "--all")
			create_image php-test

			# --line-buffered is used to produce immediate output
			# -i only since we pipe input but do not need a TTY
			# && exit 1 || [ "$?" -eq "1" ] is used to produce ok exit code on no
			# output, and error on output
			find . -type f -name '*.php' | docker run --rm --init -i \
				-v "$PWD:/app:ro" -w /app \
				"${IMAGES[php-test]}" sh -c "xargs -i sh -c \" php -l '{}' || true\"" | \
				grep -vE --line-buffered '^No syntax errors|^Errors parsing|^[[:space:]]*$' && \
				exit 1 || [ "$?" -eq "1" ]
			;;
		*)
			create_image php-test

			git ls-files '**.php' | docker run --rm --init -i \
				-v "$PWD:/app:ro" -w /app \
				"${IMAGES[php-test]}" sh -c "xargs -i sh -c \"php -l {} || true\"" | \
				grep -vE --line-buffered '^No syntax errors|^Errors parsing|^[[:space:]]*$' && \
				exit 1 || [ "$?" -eq "1" ]
			;;
	esac
}

sub_psalm() {
	create_image php-test

	debug_run_cmd docker run "${PSALM_PARAMS[@]}" "$@"
}

sub_phpunit() {
	debug_run_cmd rm -rf "$MAGENTO_FOLDER/var/cache/*"

	create_image php-test
	database_start

	debug_run_cmd docker run "${PHPUNIT_PARAMS[@]}" php vendor/bin/phpunit "$@"
}

sub_coverage() {
	debug_run_cmd rm -rf "$MAGENTO_FOLDER/var/cache/*"

	create_image php-coverage
	database_start

	debug_run_cmd docker run "${PHPUNIT_PARAMS[@]}" php -d pcov.enabled=1 -d memory_limit=2048M vendor/bin/phpunit --coverage-html coverage
}

sub_test() {
	sub_psalm
	sub_phpunit
	sub_syntax
}

sub_start() {
	nginx_start
}

sub_stop() {
	nginx_stop
	fpm_stop
	database_stop
	network_stop

	debug_run_cmd rm -rf "$MAGENTO_FOLDER/var/cache/*"
}

sub_clean() {
	sub_stop

	rm -rf .psalm .phpunit.result.cache

	debug_run_cmd docker image rm -f "${IMAGES[@]}" 2> /dev/null || true

	for v in $(docker volume ls -qf dangling=true); do
		debug_run_cmd docker volume rm "$v"
	done
}

sub_database() {
	run "database_" "$@"
}

database_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    info  Shows information about database host and port"
	echo "    query Opens up a mysql query prompt"
	echo "    start Starts the database container for testing"
	echo "    stop  Stops the database container for testing"
	echo "    run   Runs the database container interactively, useful"
	echo "          for debugging purposes"
	echo "    log   Shows the logs for database container for testing"
	echo ""
}

database_info() {
	echo "mysql://$DB_CONTAINER:3306/$DATABASE_NAME"
}

database_query() {
	network_start
	database_start

	debug_run_cmd docker exec -it "$DB_CONTAINER" mysql -D "$DATABASE_NAME"
}

database_run() {
	network_start

	debug_run_cmd docker run -it --rm --name "$DB_CONTAINER" "${DATABASE_PARAMS[@]}"
}

database_start() {
	if is_container_running "$DB_CONTAINER"; then
		debug_echo "Reusing running $DB_CONTAINER container"

		return
	fi

	network_start

	start_container "$DB_CONTAINER" "${DATABASE_PARAMS[@]}"
}

database_log() {
	debug_run_cmd docker logs "$DB_CONTAINER" "$@"
}

database_stop() {
	debug_run_cmd docker rm -f "$DB_CONTAINER" 2>/dev/null || true
}

database_allow-symlinks() {
	database_start

	debug_run_cmd docker exec -it "$DB_CONTAINER" \
		mysql -D "$DATABASE_NAME" \
		-e "INSERT INTO core_config_data (scope, scope_id, path, value) VALUES ('default', 0, 'dev/template/allow_symlink', '1') ON DUPLICATE KEY UPDATE value='1';"
}

sub_network() {
	run "network_" "$@"
}

network_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    start Starts the network interface"
	echo "    stop  Stops the network interface"
	echo ""
}

network_start() {
	if docker network inspect -f '{{.Name}}' "$NETWORK" >/dev/null 2>&1; then
		debug_echo "Reusing network $NETWORK"

		return
	fi

	debug_run_cmd docker network create -d bridge "$NETWORK" >/dev/null
}

network_stop() {
	debug_run_cmd docker network rm "$NETWORK" 2>/dev/null || true
}

sub_fpm() {
	run "fpm_" "$@"
}

fpm_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    start Starts the PHP FPM container"
	echo "    stop  Stops the PHP FPM container"
	echo "    clean Stops and removes the PHP FPM container and image"
	echo "    run   Runs the PHP FPM container interactively, useful"
	echo "          for debugging purposes"
	echo "    log   Shows the logs for the PHP FPM container"
	echo ""
}

fpm_run() {
	create_image php-fpm
	database_start
	network_start

	debug_run_cmd rm -rf "$MAGENTO_FOLDER/var/cache/*"

	debug_run_cmd docker run "${FPM_PARAMS[@]}"
}

fpm_start() {
	if is_container_running "$FPM_CONTAINER"; then
		debug_echo "Reusing running $FPM_CONTAINER container"

		return
	fi

	create_image php-fpm
	database_start
	network_start

	debug_run_cmd rm -rf "$MAGENTO_FOLDER/var/cache/*"

	start_container "$FPM_CONTAINER" "${FPM_PARAMS[@]}"
}

fpm_stop() {
	debug_run_cmd docker rm -f "$FPM_CONTAINER" 2>/dev/null || true
}

fpm_clean() {
	fpm_stop

	debug_run_cmd docker image rm -f "${IMAGES[php-fpm]}" 2> /dev/null || true
}

fpm_log() {
	debug_run_cmd docker logs "$FPM_CONTAINER" "$@"
}

sub_nginx() {
	run "nginx_" "$@"
}

nginx_help() {
	echo "Usage: $SCRIPT_NAME <subcommand> [options]"
	echo ""
	echo "Subcommands:"
	echo "    start Starts the Nginx container"
	echo "    stop  Stops the Nginx container"
	echo "    log   Shows the logs for the Nginx container"
	echo ""
}

nginx_run() {
	network_start
	fpm_start

	debug_run_cmd docker run --rm -it --name "$NGINX_CONTAINER" "${NGINX_PARAMS[@]}"
}

nginx_start() {
	if is_container_running "$NGINX_CONTAINER"; then
		debug_echo "Reusing running $NGINX_CONTAINER container"

		return
	fi

	network_start
	fpm_start

	start_container "$NGINX_CONTAINER" "${NGINX_PARAMS[@]}"
}

nginx_stop() {
	debug_run_cmd docker rm -f "$NGINX_CONTAINER" 2>/dev/null || true
}

nginx_log() {
	debug_run_cmd docker logs "$NGINX_CONTAINER" "$@"
}

sub_composer() {
	debug_run_cmd docker run "${COMPOSER_PARAMS[@]}" "$@"
}

sub_redeploy() {
	debug_run_cmd docker run "${COMPOSER_PARAMS[@]}" run-script post-install-cmd -vvv -- --redeploy
}

sub_distclean() {
	sub_clean

	rm -rf coverage vendor composer.lock
}

run() {
	local prefix=$1
	local subcommand="${2:-}"

	case "$subcommand" in
		"" | "-h" | "--help")
			"${prefix}help"
			;;

		"-v" | "--verbose")
			DEBUG="true"

			shift
			shift

			run "$prefix" "$@"

			;;

		*)
			shift
			shift

			SCRIPT_NAME="$SCRIPT_NAME $subcommand"

			if declare -f "$prefix$subcommand" > /dev/null; then
				"$prefix$subcommand" "$@"
			else
				echo "Error: '$SCRIPT_NAME' is not a known subcommand." >&2
				"${prefix}help"
				exit 1
			fi
			;;
	esac
}

run "sub_" "$@"