<?php

declare(strict_types=1);

namespace Points\Core;

use function Fixtures\loadPoints;

use Exception;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Customer;
use MageQL_Sales_Model_BuyRequest;
use Mage_Customer_Model_Group;
use Mage_Sales_Model_Quote;
use Mage_Sales_Model_Quote_Item;
use Points_Core_Helper_Data;
use Throwable;
use Varien_Object;
use Varien_Profiler;

use Points\Core\ProviderInterface;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Fixtures\Customer;
use Points\Core\Extension\Quote;

require_once __DIR__."/Base.php";
require_once __DIR__."/../Fixtures/Points.php";

class QuoteTest extends Base {
    public static function setUpBeforeClass(): void {
        loadPoints();
    }

    public function setUp(): void {
        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->clearQuotes();

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        $this->tearDown();

        throw $e;
    }

    public function createQuote(): Mage_Sales_Model_Quote {
        // Empty mock required since we have the config
        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $virtual = Mage::getModel("catalog/product");
        $virtual->setStoreId(Mage::app()->getStore()->getId())
                ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($virtual, new Varien_Object([
            "qty" => 2,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        return $quote;
    }

    public function testSetPointsEmpty(): void {
        $this->loginCustomer();

        $quote = Mage::getModel("sales/quote");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($quote)
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote->setCustomer(Mage::getSingleton("customer/session")->getCustomer());
        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quoteId = $quote->getId();

        unset($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");

        // reload quote
        $checkoutSession->setQuoteId($quoteId);

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($quoteId);

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsNotAllowed(): void {
        $this->loginCustomer();

        $quote = $this->createQuote();
        $quoteId = $quote->getId();

        unset($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");

        // reload quote
        $checkoutSession->setQuoteId($quoteId);

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(false);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($quoteId);

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsNoCustomer(): void {
        MagentoManager::init();

        $quote = $this->createQuote();
        $quoteId = $quote->getId();

        unset($quote);

        MagentoManager::reset();
        MagentoManager::init();

        $checkoutSession = Mage::getSingleton("checkout/session");

        // reload quote
        $checkoutSession->setQuoteId($quoteId);

        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPoints(): void {
        $this->loginCustomer();
        $this->createQuote();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsMaximum(): void {
        $this->loginCustomer();
        $this->createQuote();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsSimpleMaximum(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        unset($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);

        unset($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsSimpleOne(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 1, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsSimpleZero(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 0, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsSimpleMaximumLimited(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsSimpleMaximumLimitedNotIncludingTax(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(false);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsVirtualMaximum(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(99999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '{
            quote {
                items {
                    product {
                        sku
                    }
                    rowTotal { incVat exVat vat }
                    selectedPointPayment {
                        id
                        label
                        currencyRequired
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                    }
                }
                availablePointPayments {
                    id
                    label
                    points {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                        available { incVat exVat vat }
                    }
                    currency {
                        min { incVat exVat vat }
                        max { incVat exVat vat }
                        value { incVat exVat vat }
                    }
                    shipping {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                    discount {
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                    }
                }
                selectedPointPayment {
                    id
                }
                grandTotal { incVat exVat vat }
                subTotal { incVat exVat vat }
                discountTotal
                shipping {
                    method {
                        code
                        description
                    }
                    total {
                        incVat
                        exVat
                        vat
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSet(id: "TEST", points: 9999999, incVat: true) {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsToMaximumSimpleMaximum(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999999);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSetToMaximum(id: "TEST") {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsToMaximumSimpleMaximumLimited(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(true);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSetToMaximum(id: "TEST") {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testSetPointsToMaximumSimpleMaximumLimitedExVat(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "country_id" => "SE",
        ]));

        $mockProvider->method("appliesTo")
                     ->with($quote)
                     ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
                     ->with($quote->getShippingAddress())
                     ->willReturn(99);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");

        $quote->setCouponCode("PHPUnitTest1");
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(199);
        $mockProvider->method("getCustomerPointsBalanceIncludesTax")
            ->willReturn(false);

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            quotePointsSetToMaximum(id: "TEST") {
                result
                quote {
                    items {
                        product {
                            sku
                        }
                        rowTotal { incVat exVat vat }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    availablePointPayments {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                            }
                        }
                    }
                    selectedPointPayment {
                        id
                        label
                        points {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            available { incVat exVat vat }
                            selected { incVat exVat vat }
                        }
                        currency {
                            min { incVat exVat vat }
                            max { incVat exVat vat }
                            value { incVat exVat vat }
                            remaining { incVat exVat vat }
                        }
                        shipping {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                        discount {
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                    grandTotal { incVat exVat vat }
                    subTotal { incVat exVat vat }
                    discountTotal
                    shipping {
                        method {
                            code
                            description
                        }
                        total {
                            incVat
                            exVat
                            vat
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testPlaceOrder(): void {
        MagentoManager::initAdmin();

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");
        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        $setupModel->setConfigData("carriers/freeshipping/active", 1, "websites", (int)$store->getWebsiteId());
        $setupModel->setConfigData("carriers/freeshipping/free_shipping_subtotal", 0, "websites", (int)$store->getWebsiteId());

        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        /**
         * @var Quote $quote
         */
        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("freeshipping_freeshipping");
        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(9999);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->save();

        $quote->getPayment()->importData([
            "method" => "free",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        MagentoManager::reset();

        $this->loginCustomer();

        $checkoutSession = Mage::getSingleton("checkout/session");
        $mockProvider = $this->createMock(ProviderInterface::class);

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->method("getUpdatedCustomerPointsBalance")
            ->willReturn(9999999);
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceBefore")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->once())
            ->method("onOrderPlaceEnd")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"));
        $mockProvider->expects($this->never())
            ->method("onQuoteSubmitFailure");
        $mockProvider->expects($this->once())
            ->method("onQuoteSubmitAfter")
            ->with($this->isInstanceOf("Mage_Sales_Model_Order"), $this->isInstanceOf("Mage_Sales_Model_Quote"));

        Mage::register("_singleton/custom/provider", $mockProvider);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            placeOrder {
                result
                ...on PlaceOrderResultSuccess {
                    order {
                        status
                        selectedPointPayment {
                            id
                            label
                            points { incVat exVat vat }
                            currency { incVat exVat vat }
                            discount {
                                points { incVat exVat vat }
                                currency { incVat exVat vat }
                            }
                            shipping {
                                points { incVat exVat vat }
                                currency { incVat exVat vat }
                            }
                        }
                        grandTotal { incVat exVat vat }
                        subTotal { incVat exVat vat }
                        discountTotal
                        shipping {
                            method {
                                code
                                description
                            }
                            total {
                                incVat
                                exVat
                                vat
                            }
                        }
                    }
                }
            }
        }'));

        $quote = Mage::getModel("sales/quote");

        $quote->load($checkoutSession->getQuoteId());

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
        $this->assertQuoteMatchesSnapshot($quote);
    }

    public function testAddSimple(): void {
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        /**
         * @var Quote $quote
         */
        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));

        $buyRequest = MageQL_Sales_Model_BuyRequest::fromProduct($product, Mage::app()->getStore());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            mutation {
                quotePointsSet(id: "TEST", points: 9999999, incVat: true) {
                    result
                }
                addQuoteItem(buyRequest: '.json_encode($buyRequest).', qty: 3) {
                    result
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        $quoteId = Mage::getSingleton("checkout/session")->getQuoteId();

        MagentoManager::reset();
        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = Mage::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $checkoutSession->setQuoteId($quoteId);

        /**
         * @var Quote $quote
         */
        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], '
            query {
                quote {
                    subTotal {
                        incVat
                    }
                    discountTotal
                    grandTotal {
                        incVat
                    }
                    selectedPointPayment {
                        id
                        label
                    }
                    items {
                        qty
                        rowTotal { incVat }
                        product { sku }
                        selectedPointPayment {
                            id
                            label
                            currencyRequired
                            points {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                selected { incVat exVat vat }
                            }
                            currency {
                                min { incVat exVat vat }
                                max { incVat exVat vat }
                                value { incVat exVat vat }
                                remaining { incVat exVat vat }
                            }
                        }
                    }
                }
            }
        '));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
