<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use Mage_Tax_Model_Config;
use MageQL_Core_Model_Context;
use Mage_Sales_Model_Quote;

use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;
use Points\Core\Extension\Quote as QuoteExt;

class QuotePoints {
    /**
     * @var Array<QuoteAddressTotal>
     */
    protected $totals;

    /**
     * @var QuoteExt
     */
    protected $quote;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @param Array<QuoteAddressTotal> $totals
     * @param QuoteExt $quote
     */
    public function __construct(
        array $totals,
        Mage_Sales_Model_Quote $quote,
        ProviderInterface $provider
    ) {
        $this->totals = $totals;
        $this->quote = $quote;
        $this->provider = $provider;
    }

    public function getMin(): TotalInterface {
        $min = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsMin()));
        }, $this->totals));
        $tax = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return array_sum(array_map("Points\Core\Amount::tax", $t->getPointsMin()));
        }, $this->totals));

        return new QuoteItemPointsTotal(
            $min,
            $tax
        );
    }

    public function getMax(): TotalInterface {
        $max = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsMax()));
        }, $this->totals));
        $tax = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::tax", $t->getPointsMax()));
        }, $this->totals));

        return new QuoteItemPointsTotal(
            $max,
            $tax
        );
    }

    public function getValue(): TotalInterface {
        $value = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsValue()));
        }, $this->totals));
        $tax = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::tax", $t->getPointsValue()));
        }, $this->totals));

        return new QuoteItemPointsTotal(
            $value,
            $tax
        );
    }

    public function getAvailable(): TotalInterface {
        $max = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $t->getPointsMax()));
        }, $this->totals));
        $tax = (int)array_sum(array_map(function(QuoteAddressTotal $t): int {
            return (int)array_sum(array_map("Points\Core\Amount::tax", $t->getPointsMax()));
        }, $this->totals));

        $balance = $this->provider->getCustomerPointsBalance($this->quote->getStore(), $this->quote->getCustomer());
        $balanceInclTax = $this->provider->getCustomerPointsBalanceIncludesTax($this->quote->getStore(), $this->quote->getCustomer());

        $taxFrac = ($max + $tax) > 0 ? $tax / ($max + $tax) : 0;
        $balanceTax = $taxFrac > 0 ?
            ($balanceInclTax ? (int)ceil($balance * $taxFrac) : (int)floor($balance * $taxFrac / (1 + $taxFrac))) :
            0;

        $balanceExTax = $balanceInclTax ? $balance - $balanceTax : $balance;

        if($balanceExTax < $max) {
            return new QuoteItemPointsTotal(
                $balanceExTax,
                $balanceTax
            );
        }
        else {
            return new QuoteItemPointsTotal($max, $tax);
        }
    }
}
