<?php

declare(strict_types=1);

use function Points\Core\calculateRoundingSpread;
use function Points\Core\spreadCurrency;
use function Points\Core\spreadPoints;
use function Points\Core\spreadPrice;

use Points\Core\Amount;
use Points\Core\Currency;
use Points\Core\Points;
use Points\Core\Total;
use Points\Core\QuoteAddressTotal;
use Points\Core\Extension\QuoteAddress;
use Points\Core\Extension\QuoteAddressItem;
use Points\Core\Extension\Quote;
use Points\Core\ProviderInterface;

/**
 * @psalm-type Limit array{min:int, max:int}
 */
class Points_Core_Model_Sales_Total_Quote_Points extends Mage_Sales_Model_Quote_Address_Total_Abstract {
    protected $_code = "points";

    public function collect(Mage_Sales_Model_Quote_Address $address) {
        /**
         * @var QuoteAddress $address
         */
        parent::collect($address);

        $this->resetAddress($address);

        $items = $this->_getAddressItems($address);

        if(count($items) === 0) {
            // It is already reset in observer
            return $this;
        }

        $helper = Mage::helper("points_core");
        /**
         * @var Quote $quote
         */
        $quote = $address->getQuote();
        $type = $quote->getPointsType();

        if( ! $type) {
            return $this;
        }

        $store = $quote->getStore();
        $customer = $quote->getCustomer();
        $provider = $helper->getTypeProvider($store, $type);

        if( ! $provider ||
            ! $provider->appliesTo($quote) ||
            ! $customer->getId() ||
            ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            return $this;
        }

        $totals = QuoteAddressTotal::fromQuoteAddress($address, $type, $provider);
        $incVat = $quote->getPointsWantedIncludesTax() ?? (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getAmountTotal = $this->getAmountGetter($incVat);

        $pointsValue = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsTax = array_sum(array_map("Points\Core\Amount::tax", $totals->getPointsValue()));

        $pointTaxRate = $this->getPointTaxRate($totals, $incVat);
        // Tax multiplier from a total including tax
        $taxMultiplier = $pointTaxRate / (1 + $pointTaxRate);

        $requestedPoints = $this->getRequestedPoints($address, $totals, $provider, $type, $incVat);

        $included = array_filter($totals->getCurrency(), "Points\Core\Currency::isIncluded");
        $includedAmount = array_sum(array_map($getAmountTotal, $included));
        $includedTax = array_sum(array_map("Points\Core\Amount::tax", $included));

        $excluded = array_filter($totals->getCurrency(), "Points\Core\Currency::isNotIncluded");
        $excludedAmount = array_sum(array_map($getAmountTotal, $excluded));
        $excludedTax = array_sum(array_map("Points\Core\Amount::tax", $excluded));

        // Fraction of the quote which is covered by points
        $fracIncluded = ($includedAmount + $excludedAmount) > 0 ? $includedAmount / ($includedAmount + $excludedAmount) : 0;
        // Fraction of the points value to spend
        $fracPoints = $pointsValue > 0 ? $requestedPoints / $pointsValue : 0;

        $this->collectItems($address, $totals, $incVat, $requestedPoints);
        $this->collectDiscount($address, $totals, $incVat, $fracPoints);

        // addressBaseTotal includes tax
        $addressBaseTotal = array_sum($address->getAllBaseTotalAmounts());
        $baseTotalAmount = $store->roundPrice(($addressBaseTotal * $fracIncluded) * $fracPoints);
        $totalAmount = $store->convertPrice($baseTotalAmount, false);

        $baseAmount = $this->clampTax(spreadPrice([
            "value" => $baseTotalAmount * (1 - $taxMultiplier),
            "tax" => $baseTotalAmount * $taxMultiplier,
        ]), $address->getBaseTotalAmount("tax"));
        $amount = $this->clampTax(spreadPrice([
            "value" => $totalAmount * (1 - $taxMultiplier),
            "tax" => $totalAmount * $taxMultiplier,
        ]), $address->getTotalAmount("tax"));

        if($incVat) {
            $pointParts = calculateRoundingSpread([
                "value" => $requestedPoints * (1 - $taxMultiplier),
                "tax" => $requestedPoints * $taxMultiplier,
            ]);

            $address->setPointsPoints($pointParts["value"]);
            $address->setPointsTaxPoints($pointParts["tax"]);
        }
        else {
            $address->setPointsPoints($requestedPoints);
            $address->setPointsTaxPoints((int)ceil($requestedPoints * $pointTaxRate));
        }

        $address->setTotalAmount($this->getCode(), -$amount["value"]);
        $address->setBaseTotalAmount($this->getCode(), -$baseAmount["value"]);

        // Reduce tax with what we paid
        $address->addTotalAmount("tax", -$amount["tax"]);
        $address->addBaseTotalAmount("tax", -$baseAmount["tax"]);
        $address->setBasePointsTaxAmount($baseAmount["tax"]);
        $address->setPointsTaxAmount($amount["tax"]);

        $address->setPointsValue($incVat ? $pointsValue - $pointsTax : $pointsValue);
        $address->setPointsTaxValue($pointsTax);
        $address->setPointsAmountIncluded($incVat ? $includedAmount - $includedTax : $includedAmount);
        $address->setPointsTaxAmountIncluded($includedTax);
        $address->setPointsAmountExcluded($incVat ? $excludedAmount - $excludedTax : $excludedAmount);
        $address->setPointsTaxAmountExcluded($excludedTax);

        $address->setSelectedPointsTotalInstance($totals);

        $quote->setPointsWanted((int)$quote->getPointsWanted() - $requestedPoints);

        return $this;
    }

    public function fetch(Mage_Sales_Model_Quote_Address $address) {
        /**
         * @var QuoteAddress $address
         */
        $address->addTotal([
            "code"           => $this->getCode(),
            "value"          => $address->getPointsAmount(),
            "value_incl_tax" => $address->getPointsAmount() + $address->getPointsTaxAmount(),
            "value_excl_tax" => $address->getPointsAmount(),
        ]);

        return [];
    }

    /**
     * Ensures the tax value does not exceed the maximum tax on the address.
     *
     * @param float $maxTax
     */
    public function clampTax(array $data, $maxTax): array {
        $diff = $data["tax"] - $maxTax;

        if($diff > 0) {
            $data["tax"] -= $diff;
            $data["value"] += $diff;
        }

        return $data;
    }

    /**
     * @param QuoteAddress $address
     */
    private function resetAddress(
        Mage_Sales_Model_Quote_Address $address
    ): void {
        $address->setPointsValue(0);
        $address->setPointsTaxValue(0);
        $address->setPointsPoints(0);
        $address->setPointsTaxPoints(0);
        $address->setPointsAmount(0.0);
        $address->setPointsTaxAmount(0.0);
        $address->setBasePointsAmount(0.0);
        $address->setBasePointsTaxAmount(0.0);

        $address->setPointsAmountIncluded(0);
        $address->setPointsTaxAmountIncluded(0);
        $address->setPointsAmountExcluded(0);
        $address->setPointsTaxAmountExcluded(0);

        $address->setPointsShippingValue(null);
        $address->setPointsShippingTaxValue(null);
        $address->setPointsShippingPoints(0);
        $address->setPointsShippingTaxPoints(0);
        $address->setPointsShippingTotal(0.0);
        $address->setPointsShippingTaxTotal(0.0);
        $address->setBasePointsShippingTotal(0.0);
        $address->setBasePointsShippingTaxTotal(0.0);

        $address->setPointsDiscountValue(0);
        $address->setPointsDiscountTaxValue(0);
        $address->setPointsDiscountPoints(0);
        $address->setPointsDiscountTaxPoints(0);
        $address->setPointsDiscountTotal(0.0);
        $address->setPointsDiscountTaxTotal(0.0);
        $address->setBasePointsDiscountTotal(0.0);
        $address->setBasePointsDiscountTaxTotal(0.0);

        /**
         * @var QuoteAddressItem $i
         */
        foreach($this->_getAddressItems($address) as $i) {
            // Reset to make sure we do not populate if it does not have a point price
            $i->setPointsRowValue(null);
            $i->setPointsRowTaxValue(null);
            $i->setPointsRowPoints(0);
            $i->setPointsRowTaxPoints(0);
            $i->setPointsRowTotal(0.0);
            $i->setPointsRowTaxTotal(0.0);
            $i->setBasePointsRowTotal(0.0);
            $i->setBasePointsRowTaxTotal(0.0);
        }
    }

    private function getRequestedPoints(
        Mage_Sales_Model_Quote_Address $address,
        QuoteAddressTotal $totals,
        ProviderInterface $provider,
        string $type,
        bool $incVat
    ): int {
        /**
         * @var Quote $quote
         */
        $quote = $address->getQuote();
        $store = $quote->getStore();
        $customer = $quote->getCustomer();
        $getAmountTotal = $this->getAmountGetter($incVat);

        $customerPoints = $provider->getCustomerPointsBalance($store, $customer);
        $customerPointsIncludesTax = $provider->getCustomerPointsBalanceIncludesTax($store, $customer);

        $pointsValue = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsMax = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMax()));
        $pointsMin = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMin()));

        $totalLimit = Mage::getResourceModel("points_core/limit_total")->getMatchingLimit($store, $customer->getGroupId(), $type);
        $orderLimit = Mage::getResourceModel("points_core/limit_order")->getMatchingLimit($store, $customer->getGroupId(), $type);

        $pointTaxRate = $this->getPointTaxRate($totals, $incVat);
        // Tax multiplier from a total including tax
        $taxMultiplier = $pointTaxRate / (1 + $pointTaxRate);

        $availablePoints = min(
            // Reduce the available points by the tax rate if they include, otherwise increase them
            $customerPointsIncludesTax ? (
                $incVat ? $customerPoints : (int)floor($customerPoints * (1 - $taxMultiplier))
            ) : (
                $incVat ? (int)floor($customerPoints * (1 + $pointTaxRate)) : $customerPoints
            ),
            $pointsMax,
            $totalLimit ? (int)$totalLimit->getLimit() - $totalLimit->getSpentDuringLastWindow($customer) : $pointsValue,
            $orderLimit ? $orderLimit->getTotalMax($pointsValue) : $pointsValue
        );

        $minPoints = max(
            0,
            $pointsMin,
            $orderLimit ? $orderLimit->getMinValue() ?: 0 : 0
        );

        // Includes or excludes tax depending on incVat
        $requestedPoints = min($availablePoints, (int)$quote->getPointsWanted());

        // Zero it since we can't actually use any
        return $requestedPoints >= $minPoints ? $requestedPoints : 0;
    }

    /**
     * Collects itemized totals for rows and shipping.
     *
     * @param QuoteAddress $address
     */
    private function collectItems(
        Mage_Sales_Model_Quote_Address $address,
        QuoteAddressTotal $totals,
        bool $incVat,
        int $requestedPoints
    ): void {
        /**
         * @var Quote $quote
         */
        $quote = $address->getQuote();
        $store = $quote->getStore();
        $items = $this->_getAddressItems($address);
        $getAmountTotal = $this->getAmountGetter($incVat);
        $pointsValue = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsMin = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMin()));
        // Fraction of the points value to spend
        $fracPoints = $pointsValue > 0 ? $requestedPoints / $pointsValue : 0;
        // Fraction of points which is allocated to minimum
        $fracMin = $pointsValue > 0 ? $pointsMin / $pointsValue : 0;
        // Fraction of points which is on top of minimum
        $fracExMin = max($fracPoints - $fracMin, 0);
        /**
         * @var Array<string, Amount<float>>
         */
        $itemsRowPoints = [];
        /**
         * @var Array<string, Currency>
         */
        $itemsRowTotal = [];

        /**
         * @var QuoteAddressItem $i
         */
        foreach($items as $k => $i) {
            $rowValue = $totals->getRowPointsValue($i);
            $rowMin = $totals->getRowPointsMin($i) ?: new Amount(0, $incVat, 0);
            $rowCurrency = $totals->getRowCurrency($i) ?: new Currency(0, $incVat, 0, true);
            $rowValueTotal = $rowValue ? $getAmountTotal($rowValue) : 0;

            $i->setPointsRowValue($rowValue ? $rowValue->getTotalExclTax() : null);
            $i->setPointsRowTaxValue($rowValue ? $rowValue->getTax() : null);

            $rowPoints = new Amount(
                $getAmountTotal($rowMin) + $fracExMin * $rowValueTotal,
                $incVat,
                $rowMin->getTax() + ($rowValue ? $fracExMin * $rowValue->getTax() : 0)
            );

            $rowFracTotal = $rowValueTotal > 0 ? $getAmountTotal($rowPoints) / $rowValueTotal : 0;

            $itemsRowPoints["idx_$k"] = $rowPoints;
            $itemsRowTotal["idx_$k"] = new Currency(
                $rowFracTotal * $getAmountTotal($rowCurrency),
                $incVat,
                $rowFracTotal * $rowCurrency->getTax(),
                true
            );
        }

        $shippingValue = $totals->getShippingPointsValue();
        $shippingCurrency = $totals->getShippingPointsValue() ?: new Currency(0, $incVat, 0, true);

        if($shippingValue) {
            $address->setPointsShippingValue($shippingValue->getTotalExclTax());
            $address->setPointsShippingTaxValue($shippingValue->getTax());

            $shippingValueTotal = $getAmountTotal($shippingValue);
            $shippingPoints = new Amount(
                $fracExMin * $shippingValueTotal,
                $incVat,
                $fracExMin * $shippingValue->getTax()
            );

            $shippingFracTotal = $shippingValueTotal > 0 ? $getAmountTotal($shippingPoints) / $shippingValueTotal : 0;

            $itemsRowPoints["shipping"] = $shippingPoints;
            $itemsRowTotal["shipping"] = new Currency(
                $shippingFracTotal * $getAmountTotal($shippingCurrency),
                $incVat,
                $shippingFracTotal * $shippingCurrency->getTax(),
                true
            );
        }

        $itemsRowPointsSpread = spreadPoints($itemsRowPoints);
        $itemsRowTotalSpread = spreadCurrency($itemsRowTotal);

        /**
         * @var QuoteAddressItem $i
         */
        foreach($items as $k => $i) {
            $rowPoints = $itemsRowPointsSpread["idx_$k"] ?? new Points(0, $incVat, 0);
            $rowTotal = $itemsRowTotalSpread["idx_$k"] ?? new Currency(0, $incVat, 0, true);

            $i->setPointsRowPoints($rowPoints->getTotalExclTax());
            $i->setPointsRowTaxPoints($rowPoints->getTax());
            $i->setPointsRowTotal($rowTotal->getTotalExclTax());
            $i->setPointsRowTaxTotal($rowTotal->getTax());
            $i->setBasePointsRowTotal($store->convertPrice($rowTotal->getTotalExclTax(), false));
            $i->setBasePointsRowTaxTotal($store->convertPrice($rowTotal->getTax(), false));
        }

        if(array_key_exists("shipping", $itemsRowPointsSpread)) {
            $shippingPoints = $itemsRowPointsSpread["shipping"];

            $quote->setPointsShippingPoints($shippingPoints->getTotalExclTax());
            $quote->setPointsShippingTaxPoints($shippingPoints->getTax());
        }

        if(array_key_exists("shipping", $itemsRowTotalSpread)) {
            $shippingTotal = $itemsRowTotalSpread["shipping"];

            $quote->setPointsShippingTotal($shippingTotal->getTotalExclTax());
            $quote->setPointsShippingTaxTotal($shippingTotal->getTax());
            $quote->setBasePointsShippingTotal($store->convertPrice($shippingTotal->getTotalExclTax(), false));
            $quote->setBasePointsShippingTaxTotal($store->convertPrice($shippingTotal->getTax(), false));
        }
    }

    /**
     * Collects discount total in points.
     *
     * @param QuoteAddress $address
     */
    private function collectDiscount(
        Mage_Sales_Model_Quote_Address $address,
        QuoteAddressTotal $totals,
        bool $incVat,
        float $fracPoints
    ): void {
        $quote = $address->getQuote();
        $store = $quote->getStore();

        $pointTaxRate = $this->getPointTaxRate($totals, $incVat);
        // Tax multiplier from a total including tax
        $taxMultiplier = $pointTaxRate / (1 + $pointTaxRate);

        $includedDiscount = array_filter($totals->getDiscount(), "Points\Core\Currency::isIncluded");
        $baseDiscountAmount = array_sum(array_map("Points\Core\Amount::totalInclTax", $includedDiscount));

        $pointsDiscount = (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $totals->getPointsDiscount()));
        $pointsTaxDiscount = (int)array_sum(array_map("Points\Core\Amount::tax", $totals->getPointsDiscount()));

        $baseTotalAmount = $store->roundPrice($baseDiscountAmount * $fracPoints);
        $totalAmount = $store->convertPrice($baseTotalAmount, false);
        $pointDiscountParts = calculateRoundingSpread([
            "value" => $fracPoints * $pointsDiscount,
            "tax" => $fracPoints * $pointsTaxDiscount,
        ]);

        $baseAmount = spreadPrice([
            "value" => $baseTotalAmount * (1 - $taxMultiplier),
            "tax" => $baseTotalAmount * $taxMultiplier,
        ]);
        $amount = spreadPrice([
            "value" => $totalAmount * (1 - $taxMultiplier),
            "tax" => $totalAmount * $taxMultiplier,
        ]);

        $address->setPointsDiscountPoints($pointDiscountParts["value"]);
        $address->setPointsDiscountTaxPoints($pointDiscountParts["tax"]);
        $address->setPointsDiscountValue($pointsDiscount);
        $address->setPointsDiscountTaxValue($pointsTaxDiscount);
        $address->setPointsDiscountTotal($baseAmount["value"]);
        $address->setPointsDiscountTaxTotal($baseAmount["tax"]);
        $address->setBasePointsDiscountTotal($amount["value"]);
        $address->setBasePointsDiscountTaxTotal($amount["tax"]);
    }

    private function getPointTaxRate(
        QuoteAddressTotal $totals,
        bool $incVat
    ): float {
        $getAmountTotal = $this->getAmountGetter($incVat);

        $pointsValue = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsTax = array_sum(array_map("Points\Core\Amount::tax", $totals->getPointsValue()));

        // Rate from a total without tax
        $pointsValueExclTax = $incVat ? $pointsValue - $pointsTax : $pointsValue;

        return $pointsValueExclTax > 0 ? $pointsTax / $pointsValueExclTax : 0;
    }

    private function getAmountGetter(bool $incVat): callable {
        return $incVat ?
            "Points\Core\Amount::totalInclTax" :
            "Points\Core\Amount::totalExclTax";
    }
}
