<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use Mage;
use Mage_Sales_Model_Quote;
use Points_Core_Model_Limit_Order;
use Points_Core_Model_Limit_Total;

use Points\Core\Amount;
use Points\Core\ProviderInterface;
use Points\Core\Total\QuoteAddress;

// TODO: Should probably be moved to QuoteAddress
class QuotePoints {
    /**
     * @var string
     */
    protected $type;

    /**
     * @var QuoteAddress
     */
    protected $total;

    /**
     * @var Mage_Sales_Model_Quote
     */
    protected $quote;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @var Points_Core_Model_Limit_Order|null|false
     */
    private $orderLimit = null;

    /**
     * @var Points_Core_Model_Limit_Total|null|false
     */
    private $totalLimit = null;

    public function __construct(
        string $type,
        QuoteAddress $total,
        Mage_Sales_Model_Quote $quote,
        ProviderInterface $provider
    ) {
        $this->type = $type;
        $this->total = $total;
        $this->quote = $quote;
        $this->provider = $provider;
    }

    public function getMin(): PointsTotal {
        $points = $this->total->getPoints();
        $value = $points->getValue();
        $min = $points->getMin();
        $orderLimit = $this->getOrderLimit();

        if($orderLimit) {
            $valueTotal = $value->getTotalAndTax((bool)$orderLimit->getIncludesTax());
            $orderLimitMin = $orderLimit->getTotalMin($valueTotal);
            $minTotal = $min->getTotalAndTax((bool)$orderLimit->getIncludesTax());

            if($orderLimitMin > $minTotal) {
                return new PointsTotal($value->multiply($valueTotal > 0 ? $orderLimitMin / $valueTotal : 0, 0));
            }
        }

        return new PointsTotal($min);
    }

    public function getMax(): PointsTotal {
        return new PointsTotal($this->getMaxInner());
    }

    public function getValue(): PointsTotal {
        return new PointsTotal($this->total->getPoints()->getValue());
    }

    public function getAvailable(): PointsTotal {
        $totalLimit = $this->getTotalLimit();
        $max = $this->getMaxInner();

        $balance = $this->provider->getCustomerPointsBalance($this->quote->getStore(), $this->quote->getCustomer());
        $balanceInclTax = $this->provider->getCustomerPointsBalanceIncludesTax($this->quote->getStore(), $this->quote->getCustomer());

        $remaining = $max;

        if($totalLimit) {
            $customerRemaining = $totalLimit->getRemaining($this->quote->getCustomer());

            if($customerRemaining !== null) {
                $maxTotal = $max->getTotalAndTax((bool)$totalLimit->getIncludesTax());

                $remaining = $remaining->multiply($maxTotal > 0 ? $customerRemaining / $maxTotal : 0, 0);
            }
        }

        $remainingTotal = $remaining->getTotalAndTax($balanceInclTax);

        if($balance < $remainingTotal) {
            $remaining = $remaining->multiply($remainingTotal > 0 ? $balance / $remainingTotal : 0, 0);
        }

        return new PointsTotal($remaining);
    }

    public function getOrderLimit(): ?Points_Core_Model_Limit_Order {
        if($this->orderLimit === null) {
            $this->orderLimit = Mage::getResourceModel("points_core/limit_order")
                ->getMatchingLimit($this->quote->getStore(), $this->quote->getCustomerGroupId(), $this->type) ?: false;
        }

        return $this->orderLimit ?: null;
    }

    public function getTotalLimit(): ?Points_Core_Model_Limit_Total {
        if($this->totalLimit === null) {
            $this->totalLimit = Mage::getResourceModel("points_core/limit_total")
                ->getMatchingLimits($this->quote->getStore(), $this->quote->getCustomerGroupId(), [$this->type])[$this->type] ?? false;
        }

        return $this->totalLimit ?: null;
    }

    // TODO: Move to QuoteAddress or something
    /**
     * @return Amount<int>
     */
    private function getMaxInner(): Amount {
        $points = $this->total->getPoints();
        $value = $points->getValue();
        $max = $points->getMax();

        $orderLimit = $this->getOrderLimit();

        if($orderLimit) {
            $valueTotal = $value->getTotalAndTax((bool)$orderLimit->getIncludesTax());
            $maxTotal = $max->getTotalAndTax((bool)$orderLimit->getIncludesTax());
            $orderMax = $orderLimit->getTotalMax($valueTotal);

            if($orderMax < $maxTotal) {
                return $value->multiply($valueTotal > 0 ? $orderMax / $valueTotal : 0, 0);
            }
        }

        return $max;
    }
}
