<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use Mage_Sales_Model_Quote;
use MageQL\Context;
use Points\Core\ProviderInterface;
use Points\Core\Total\QuoteAddress;
use Points\Core\Total\Shipping as ShippingTotal;

class Quote {
    /**
     * @var string
     */
    protected $type;

    /**
     * @var ProviderInterface
     */
    protected $provider;

    /**
     * @var Mage_Sales_Model_Quote
     */
    protected $quote;

    /**
     * @var QuoteAddress
     */
    protected $total;

    /**
     * @var ?QuotePoints
     */
    private $points;

    public function __construct(
        string $type,
        ProviderInterface $provider,
        Mage_Sales_Model_Quote $quote,
        QuoteAddress $total
    ) {
        $this->type = $type;
        $this->provider = $provider;
        $this->quote = $quote;
        $this->total = $total;
    }

    public function getId(): string {
        return $this->type;
    }

    public function getLabel(array $args, Context $ctx): string {
        return $this->provider->getLabel($ctx->getStore());
    }

    /**
     * @return Array<QuoteRejectionReason>
     */
    public function getRejectionReasons(array $unusedArgs, Context $ctx): array {
        $reasons = [];

        $balance = $this->provider->getCustomerPointsBalance($this->quote->getStore(), $this->quote->getCustomer());
        $balanceInclTax = $this->provider->getCustomerPointsBalanceIncludesTax($this->quote->getStore(), $this->quote->getCustomer());
        $points = $this->getPoints();
        $minPoints = $points->getMin();
        $totalLimit = $points->getTotalLimit();
        $orderLimit = $points->getOrderLimit();

        $requestedBalance = $balanceInclTax ? $minPoints->getIncVat($ctx) : $minPoints->getExVat($ctx);

        if($balance < $requestedBalance) {
            $reasons[] = new QuoteRejectionReasonBalance(
                $requestedBalance,
                $balanceInclTax,
                $balance
            );
        }

        if($totalLimit) {
            $remaining = $totalLimit->getRemaining($this->quote->getCustomer());
            $requestedTotal = $totalLimit->getIncludesTax() ? $minPoints->getIncVat($ctx) : $minPoints->getExVat($ctx);

            if($remaining !== null && $remaining < $requestedTotal) {
                $reasons[] = new QuoteRejectionReasonTotalLimit(
                    $requestedTotal,
                    $totalLimit,
                    $remaining
                );
            }
        }

        if($orderLimit) {
            $quoteValue = $this->total->getPoints()->getValue()->getTotalAndTax((bool)$orderLimit->getIncludesTax());
            $orderLimitMax = $orderLimit->getTotalMax($quoteValue);
            $requestedOrder = $orderLimit->getIncludesTax() ? $minPoints->getIncVat($ctx) : $minPoints->getExVat($ctx);

            if($orderLimitMax < $requestedOrder) {
                $reasons[] = new QuoteRejectionReasonOrderLimit(
                    $requestedOrder,
                    (bool)$orderLimit->getIncludesTax(),
                    $orderLimitMax
                );
            }
        }

        return $reasons;
    }

    public function getCurrencyRequired(): bool {
        $excluded = $this->total->getExcluded()->getTotal()->getTotalInclTax();

        if($excluded > 0) {
            return true;
        }

        $points = $this->total->getPoints();

        return $points->getMax()->getTotalInclTax() < $points->getValue()->getTotalInclTax();
    }

    public function getPoints(): QuotePoints {
        if( ! $this->points) {
            $this->points = new QuotePoints($this->type, $this->total, $this->quote, $this->provider);
        }

        return $this->points;
    }

    public function getCurrency(): QuoteCurrency {
        return new QuoteCurrency($this->total, $this->getPoints());
    }

    public function getShipping(): ?QuoteShipping {
        foreach($this->total->getTotals() as $total) {
            if($total instanceof ShippingTotal) {
                return new QuoteShipping($total);
            }
        }

        return null;
    }

    public function getDiscount(): ?QuoteDiscount {
        $discount = $this->total->getDiscountTotal();

        if($discount->getTotalInclTax() > 0) {
            return new QuoteDiscount($this->total, $this->getPoints());
        }

        return null;
    }
}
