<?php

declare(strict_types=1);

namespace Points\Core\Issues;

use function Fixtures\loadPoints;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Fixtures\TaxRates;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Group;
use Points\Core\Base;
use Points\Core\ProviderInterface;
use Throwable;
use Varien_Object;

require_once __DIR__."/../Base.php";
require_once __DIR__."/../../Fixtures/Points.php";
require_once __DIR__."/../../../vendor/crossroads/mageql-catalog/test/Fixtures/TaxRates.php";

/**
 * Not a completely 100% coverage of this issue, but it seems to arise when the following happens:
 *
 * * Configurable product
 * * The configurable product does not have tax
 * * Minimum amount of points for the configurable product is equal to its cost
 * * Another product with points payment is present
 */
class Gen682Test extends Base {
    public function setUp(): void {
        loadPoints();

        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->clearQuotes();

        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $config = Mage::getModel("catalog/product");
        $config1 = Mage::getModel("catalog/product");
        $configPrice1 = Mage::getModel("points_core/product_price");

        $config->setStoreId($store->getId());
        $config->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));
        $config->addData([
            "store_id" => $store->getId(),
            "tax_class_id" => 0,
        ]);
        $config->save();

        $config1->setStoreId($store->getId());
        $config1->load(Mage::getModel("catalog/product")->getIdBySku("test-config-child-1"));
        $config1->addData([
            "store_id" => $store->getId(),
            "tax_class_id" => 0,
        ]);
        $config1->save();

        $configPrice1->loadByStoreProductTypeCustomerGroupId($store, $config1, "TEST", Mage_Customer_Model_Group::NOT_LOGGED_IN_ID);
        $configPrice1->addData([
            "store_id" => $store->getId(),
            "product_id" => $config1->getId(),
            "customer_group_id" => Mage_Customer_Model_Group::NOT_LOGGED_IN_ID,
            "type" => "TEST",
            "price" => 1000,
            "min_price" => 1000,
            "max_price" => null,
        ]);
        $configPrice1->save();

        Mage::getSingleton("points_core/product_index")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();

        MagentoManager::reset();
        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $config = Mage::getModel("catalog/product");
        $config1 = Mage::getModel("catalog/product");

        $config->setStoreId($store->getId());
        $config->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));
        $config->addData([
            "store_id" => $store->getId(),
            "tax_class_id" => TaxRates::getTaxClass(TaxRates::PRODUCT_TAX_CLASS)->getId(),
        ]);
        $config->save();

        $config1->setStoreId($store->getId());
        $config1->load(Mage::getModel("catalog/product")->getIdBySku("test-config-child-1"));
        $config1->addData([
            "store_id" => $store->getId(),
            "tax_class_id" => TaxRates::getTaxClass(TaxRates::PRODUCT_TAX_CLASS)->getId(),
        ]);
        $config1->save();

        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();

        MagentoManager::reset();

        loadPoints();

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    public function testConfigurable(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");

        $this->assertNotFalse($attr);

        $attrOptions = $attr->getSource()->getAllOptions();
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Red":
                $attrBlack = $o["value"];
                break;
            }
        }

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-config"));

        $quote->addProduct($product, new Varien_Object([
            "qty" => 1,
            "super_attribute" => [$attr->getId() => $attrBlack],
        ]));

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
               ->load(Mage::getModel("catalog/product")->getIdBySku("test-simple"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->getShippingAddress()->setShippingMethod("flatrate_flatrate");
        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(1000);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);
    }
}
