<?php

declare(strict_types=1);

namespace Points\Core\Issues;

use function Fixtures\loadPoints;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Fixtures\TaxRates;
use Mage;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Group;
use Mage_Tax_Model_Class;
use Points\Core\Base;
use Points\Core\Extension\Quote;
use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;
use Throwable;
use Varien_Object;

require_once __DIR__."/../Base.php";
require_once __DIR__."/../../Fixtures/Points.php";
require_once __DIR__."/../../../vendor/crossroads/mageql-catalog/test/Fixtures/TaxRates.php";

/**
 * This issue is a rounding issue for the following conditions:
 *
 *  * points value: 8000
 *  * currency value: 29 kr
 *  * tax: 12%
 *  * qty: 2 * N
 *  * points only
 *  * pay all
 */
class Gen1901Test extends Base {
    public function setUp(): void {
        loadPoints();

        Config::setConfigPath("default/points/providers/TEST/model", "custom/provider");

        $this->clearQuotes();

        MagentoManager::initAdmin();

        $taxRates = new TaxRates();

        $customerClass = $taxRates->createClass(TaxRates::CUSTOMER_TAX_CLASS, Mage_Tax_Model_Class::TAX_CLASS_TYPE_CUSTOMER);
        $productClass = $taxRates->createClass("Gen1901 12% tax rate", Mage_Tax_Model_Class::TAX_CLASS_TYPE_PRODUCT);
        $rate = $taxRates->createRate("SE", "Gen1901 12% SE", 12.00);

        $taxRates->createRule("Gen1901", $customerClass, $productClass, $rate);

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $virtual = Mage::getModel("catalog/product");
        $virtualPrice = Mage::getModel("points_core/product_price");

        $virtual->setStoreId($store->getId());
        $virtual->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $virtual->addData([
            "price" => 29.0,
            "store_id" => $store->getId(),
            "tax_class_id" => $productClass->getId(),
        ]);
        $virtual->save();

        $virtualPrice->loadByStoreProductTypeCustomerGroupId($store, $virtual, "TEST", Mage_Customer_Model_Group::NOT_LOGGED_IN_ID);
        $virtualPrice->addData([
            "store_id" => $store->getId(),
            "product_id" => $virtual->getId(),
            "customer_group_id" => Mage_Customer_Model_Group::NOT_LOGGED_IN_ID,
            "type" => "TEST",
            "price" => 8000,
            "min_price" => 1000,
            "max_price" => null,
        ]);
        $virtualPrice->save();

        Mage::getSingleton("points_core/product_index")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();

        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigPath("default/points/providers/TEST/model");

        $this->clearQuotes();

        MagentoManager::reset();
        MagentoManager::initAdmin();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);
        $virtual = Mage::getModel("catalog/product");

        $virtual->setStoreId($store->getId());
        $virtual->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $virtual->addData([
            "price" => 9.99,
            "store_id" => $store->getId(),
            "tax_class_id" => TaxRates::getTaxClass(TaxRates::PRODUCT_TAX_CLASS)->getId(),
        ]);
        $virtual->save();

        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();

        MagentoManager::reset();

        loadPoints();

        MagentoManager::reset();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();

        throw $t;
    }

    public function testVirtualTwo(): void {
        MagentoManager::reset();

        $this->loginCustomer();

        $mockProvider = $this->createMock(ProviderInterface::class);
        $checkoutSession = Mage::getSingleton("checkout/session");
        $customerSession = MAge::getSingleton("customer/session");

        Mage::register("_singleton/custom/provider", $mockProvider);

        /**
         * @var Quote $quote
         */
        $quote = $checkoutSession->getQuote();

        if($customerSession->isLoggedIn()) {
            $quote->setCustomer($customerSession->getCustomer());
        }

        $quote->setStoreId(Mage::app()->getStore()->getId());
        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));
        $quote->setShippingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $mockProvider->method("appliesTo")
            ->with($checkoutSession->getQuote())
            ->willReturn(true);
        $mockProvider->method("getLabel")
            ->willReturn("Test Points");
        $mockProvider->method("getCustomerRedemptionAllowed")
            ->willReturn(true);
        $mockProvider->method("getQuoteShippingPrice")
            ->willReturn(100);
        $mockProvider->method("getCustomerPointsBalance")
            ->willReturn(9999999);

        $product = Mage::getModel("catalog/product");
        $product->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-virtual"));
        $quote->addProduct($product, new Varien_Object([
            "qty" => 2,
        ]));

        $quote->setCustomerEmail("test@example.com");

        $quote->setPointsType("TEST");
        $quote->setPointsWanted(8000 * 2);
        $quote->setPointsWantedIncludesTax(true);
        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $this->assertQuoteMatchesSnapshot($quote);
    }
}
