<?php

declare(strict_types=1);

namespace Points\Core\Schema;

use MageQL_Core_Model_Price_Abstract;
use MageQL_Core_Model_Context;
use Mage_Tax_Model_Config;

use Points\Core\Currency;
use Points\Core\QuoteAddressTotal;

class QuoteCurrency {
    /**
     * @var QuoteAddressTotal
     */
    protected $total;

    /**
     * @var QuotePoints
     */
    protected $points;

    public function __construct(
        QuoteAddressTotal $total,
        QuotePoints $points
    ) {
        $this->total = $total;
        $this->points = $points;
    }

    public function getMin(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Price_Abstract {
        $incVat = (bool)$ctx->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getTotal = $incVat ? "Points\Core\Amount::totalInclTax" : "Points\Core\Amount::totalExclTax";

        $value = $this->points->getValue()->getIncVat($ctx);
        $max = $this->points->getMax()->getIncVat($ctx);

        $rate = $value > 0 ? max(1 - $max / $value, 0) : 0;

        return new QuoteCurrencyTotal(
            $this->getExcluded($getTotal) + $this->getIncluded($getTotal) * $rate,
            $incVat,
            $this->getExcludedTax() + $this->getIncludedTax() * $rate
        );
    }

    public function getMax(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Price_Abstract {
        $incVat = (bool)$ctx->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getTotal = $incVat ? "Points\Core\Amount::totalInclTax" : "Points\Core\Amount::totalExclTax";

        $value = $this->points->getValue()->getIncVat($ctx);
        $min = $this->points->getMin()->getIncVat($ctx);

        $rate = $value > 0 ? (1 - max($min / $value, 0)) : 0;

        return new QuoteCurrencyTotal(
            $this->getExcluded($getTotal) + $this->getIncluded($getTotal) * $rate,
            $incVat,
            $this->getExcludedTax() + $this->getIncludedTax() * $rate
        );
    }

    public function getValue(
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Price_Abstract {
        $incVat = (bool)$ctx->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getTotal = $incVat ? "Points\Core\Amount::totalInclTax" : "Points\Core\Amount::totalExclTax";

        return new QuoteCurrencyTotal(
            $this->getExcluded($getTotal) + $this->getIncluded($getTotal),
            $incVat,
            $this->getExcludedTax() + $this->getIncludedTax()
        );
    }

    /**
     * @param callable(Currency):float $getTotal
     */
    protected function getIncluded(callable $getTotal): float {
        return array_sum(array_map($getTotal, array_filter($this->total->getCurrency(), "Points\Core\Currency::isIncluded")));
    }

    protected function getIncludedTax(): float {
        return array_sum(array_map("Points\Core\Amount::tax", array_filter($this->total->getCurrency(), "Points\Core\Currency::isIncluded")));
    }

    /**
     * @param callable(Currency):float $getTotal
     */
    protected function getExcluded(callable $getTotal): float {
        return array_sum(array_map($getTotal, array_filter($this->total->getCurrency(), "Points\Core\Currency::isNotIncluded")));
    }

    protected function getExcludedTax(): float {
        return array_sum(array_map("Points\Core\Amount::tax", array_filter($this->total->getCurrency(), "Points\Core\Currency::isNotIncluded")));
    }
}
