<?php

declare(strict_types=1);

use function Points\Core\calculateRoundingSpread;
use function Points\Core\spreadCurrency;
use function Points\Core\spreadPoints;
use function Points\Core\spreadPrice;

use Points\Core\Amount;
use Points\Core\Currency;
use Points\Core\Points;
use Points\Core\Total;
use Points\Core\QuoteAddressTotal;
use Points\Core\Extension\QuoteAddress;
use Points\Core\Extension\QuoteAddressItem;
use Points\Core\Extension\Quote;
use Points\Core\ProviderInterface;

/**
 * @psalm-type Limit array{min:int, max:int}
 */
class Points_Core_Model_Sales_Total_Quote_Points extends Mage_Sales_Model_Quote_Address_Total_Abstract {
    protected $_code = "points";

    public function collect(Mage_Sales_Model_Quote_Address $address) {
        /**
         * @var QuoteAddress $address
         */
        parent::collect($address);

        $this->resetAddress($address);

        $items = $this->_getAddressItems($address);

        if(count($items) === 0) {
            // It is already reset in observer
            return $this;
        }

        $helper = Mage::helper("points_core");
        /**
         * @var Quote $quote
         */
        $quote = $address->getQuote();
        $type = $quote->getPointsType();

        if( ! $type) {
            return $this;
        }

        $store = $quote->getStore();
        $customer = $quote->getCustomer();
        $provider = $helper->getTypeProvider($type);

        if( ! $provider ||
            ! $provider->appliesTo($quote) ||
            ! $customer->getId() ||
            ! $provider->getCustomerRedemptionAllowed($store, $customer)) {
            return $this;
        }

        $totals = QuoteAddressTotal::fromQuoteAddress($address, $type, $provider);
        $incVat = $quote->getPointsWantedIncludesTax() ?? (bool)$store->getConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX);
        $getAmountTotal = $this->getAmountGetter($incVat);

        $pointsValue = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsTax = array_sum(array_map("Points\Core\Amount::tax", $totals->getPointsValue()));

        $pointTaxRate = $this->getPointTaxRate($totals, $incVat);
        // Tax multiplier from a total including tax
        $taxMultiplier = $pointTaxRate / (1 + $pointTaxRate);

        $requestedPoints = $this->getRequestedPoints($address, $totals, $provider, $type, $incVat);

        $included = array_filter($totals->getCurrency(), "Points\Core\Currency::isIncluded");
        $includedAmount = array_sum(array_map($getAmountTotal, $included));
        $includedTax = array_sum(array_map("Points\Core\Amount::tax", $included));

        $excluded = array_filter($totals->getCurrency(), "Points\Core\Currency::isNotIncluded");
        $excludedAmount = array_sum(array_map($getAmountTotal, $excluded));
        $excludedTax = array_sum(array_map("Points\Core\Amount::tax", $excluded));

        $includedTotal = $incVat ? $includedAmount : $includedAmount + $includedTax;
        $excludedTotal = $incVat ? $excludedAmount : $excludedAmount + $excludedTax;

        // Fraction of the points value to spend
        $fracPoints = $pointsValue > 0 ? $requestedPoints / $pointsValue : 0;

        $this->collectItems($address, $totals, $incVat, $requestedPoints);
        $this->collectDiscount($address, $totals, $incVat, $fracPoints);

        // addressBaseTotal includes tax
        $addressBaseTotal = array_sum($address->getAllBaseTotalAmounts());

        // Sanity check
        if(abs($includedTotal + $excludedTotal - $addressBaseTotal) >= 0.01) {
            $message = sprintf(
                "%s: Mismatch for quote (%d) address (%d) base total (%f) vs included (%f) + excluded (%f)",
                __METHOD__,
                $quote->getId(),
                $address->getId(),
                $addressBaseTotal,
                $includedTotal,
                $excludedTotal
            );

            Mage::log($message, Zend_Log::WARN);

            if(Mage::getIsDeveloperMode()) {
                throw new Exception($message);
            }
        }

        $baseTotalAmount = $store->roundPrice($includedTotal * $fracPoints);
        $totalAmount = $store->convertPrice($baseTotalAmount, false);

        if($incVat) {
            $pointParts = calculateRoundingSpread([
                "value" => $requestedPoints * (1 - $taxMultiplier),
                "tax" => $requestedPoints * $taxMultiplier,
            ]);

            $address->setPointsPoints($pointParts["value"]);
            $address->setPointsTaxPoints($pointParts["tax"]);
        }
        else {
            $address->setPointsPoints($requestedPoints);
            $address->setPointsTaxPoints((int)ceil($requestedPoints * $pointTaxRate));
        }

        $baseTaxAmount = $baseTotalAmount * $taxMultiplier;
        $taxAmount = $totalAmount * $taxMultiplier;
        $baseAmount = spreadPrice($this->clampTax([
            "value" => $baseTotalAmount - $baseTaxAmount,
            "tax" => $baseTaxAmount,
        ], $address->getBaseTotalAmount("tax")));
        $amount = spreadPrice($this->clampTax([
            "value" => $totalAmount - $taxAmount,
            "tax" => $taxAmount,
        ], $address->getTotalAmount("tax")));

        $address->setTotalAmount($this->getCode(), -$amount["value"]);
        $address->setBaseTotalAmount($this->getCode(), -$baseAmount["value"]);

        // Reduce tax with what we paid
        $address->addTotalAmount("tax", -$amount["tax"]);
        $address->addBaseTotalAmount("tax", -$baseAmount["tax"]);
        $address->setBasePointsTaxAmount($baseAmount["tax"]);
        $address->setPointsTaxAmount($amount["tax"]);

        $address->setPointsValue($incVat ? $pointsValue - $pointsTax : $pointsValue);
        $address->setPointsTaxValue($pointsTax);
        $address->setPointsAmountIncluded($incVat ? $includedAmount - $includedTax : $includedAmount);
        $address->setPointsTaxAmountIncluded($includedTax);
        $address->setPointsAmountExcluded($incVat ? $excludedAmount - $excludedTax : $excludedAmount);
        $address->setPointsTaxAmountExcluded($excludedTax);

        $quote->setPointsWanted((int)$quote->getPointsWanted() - $requestedPoints);

        return $this;
    }

    public function fetch(Mage_Sales_Model_Quote_Address $address) {
        /**
         * @var QuoteAddress $address
         */
        $address->addTotal([
            "code"           => $this->getCode(),
            "value"          => $address->getPointsAmount(),
            "value_incl_tax" => $address->getPointsAmount() + $address->getPointsTaxAmount(),
            "value_excl_tax" => $address->getPointsAmount(),
        ]);

        return [];
    }

    /**
     * Ensures the tax value does not exceed the maximum tax on the address.
     *
     * @param float $maxTax
     */
    public function clampTax(array $data, $maxTax): array {
        $diff = $data["tax"] - $maxTax;

        if($diff > 0) {
            $data["tax"] -= $diff;
            $data["value"] += $diff;
        }

        return $data;
    }

    /**
     * @param QuoteAddress $address
     */
    private function resetAddress(
        Mage_Sales_Model_Quote_Address $address
    ): void {
        $address->setPointsValue(0);
        $address->setPointsTaxValue(0);
        $address->setPointsPoints(0);
        $address->setPointsTaxPoints(0);
        $address->setPointsAmount(0.0);
        $address->setPointsTaxAmount(0.0);
        $address->setBasePointsAmount(0.0);
        $address->setBasePointsTaxAmount(0.0);

        $address->setPointsAmountIncluded(0);
        $address->setPointsTaxAmountIncluded(0);
        $address->setPointsAmountExcluded(0);
        $address->setPointsTaxAmountExcluded(0);

        $address->setPointsShippingValue(null);
        $address->setPointsShippingTaxValue(null);
        $address->setPointsShippingPoints(0);
        $address->setPointsShippingTaxPoints(0);
        $address->setPointsShippingTotal(0.0);
        $address->setPointsShippingTaxTotal(0.0);
        $address->setBasePointsShippingTotal(0.0);
        $address->setBasePointsShippingTaxTotal(0.0);

        $address->setPointsDiscountValue(0);
        $address->setPointsDiscountTaxValue(0);
        $address->setPointsDiscountPoints(0);
        $address->setPointsDiscountTaxPoints(0);
        $address->setPointsDiscountTotal(0.0);
        $address->setPointsDiscountTaxTotal(0.0);
        $address->setBasePointsDiscountTotal(0.0);
        $address->setBasePointsDiscountTaxTotal(0.0);

        /**
         * @var QuoteAddressItem $i
         */
        foreach($this->_getAddressItems($address) as $i) {
            // Reset to make sure we do not populate if it does not have a point price
            $i->setPointsRowValue(null);
            $i->setPointsRowTaxValue(null);
            $i->setPointsRowPoints(0);
            $i->setPointsRowTaxPoints(0);
            $i->setPointsRowTotal(0.0);
            $i->setPointsRowTaxTotal(0.0);
            $i->setBasePointsRowTotal(0.0);
            $i->setBasePointsRowTaxTotal(0.0);
        }
    }

    private function getRequestedPoints(
        Mage_Sales_Model_Quote_Address $address,
        QuoteAddressTotal $totals,
        ProviderInterface $provider,
        string $type,
        bool $incVat
    ): int {
        /**
         * @var Quote $quote
         */
        $quote = $address->getQuote();
        $store = $quote->getStore();
        $customer = $quote->getCustomer();
        $getAmountTotal = $this->getAmountGetter($incVat);

        $customerPoints = $provider->getCustomerPointsBalance($store, $customer);
        $customerPointsIncludesTax = $provider->getCustomerPointsBalanceIncludesTax($store, $customer);

        $pointsValue = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsMax = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMax()));
        $pointsMin = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMin()));

        $totalLimit = Mage::getResourceModel("points_core/limit_total")->getMatchingLimit($store, $customer->getGroupId(), $type);
        $orderLimit = Mage::getResourceModel("points_core/limit_order")->getMatchingLimit($store, $customer->getGroupId(), $type);

        // TODO: Refactor to be reusable in QuotePoints
        $pointTaxRate = $this->getPointTaxRate($totals, $incVat);
        $convertTax =
            /**
             * @param int $value
             * @param bool $valueIncVat
             * @return int
             */
            function($value, $valueIncVat) use($incVat, $pointTaxRate) {
                if($valueIncVat) {
                    // Tax multiplier from a total including tax
                    $taxMultiplier = $pointTaxRate / (1 + $pointTaxRate);

                    return $incVat ? $value : (int)floor($value * (1 - $taxMultiplier));
                }

                return $incVat ? (int)floor($value * (1 + $pointTaxRate)) : $value;
            };

        $totalLimitPoints = $pointsValue;

        if($totalLimit) {
            $totalLimitRemaining = $totalLimit->getRemaining($customer);

            if($totalLimitRemaining !== null) {
                $totalLimitPoints = $convertTax(
                    $totalLimitRemaining,
                    $totalLimit->getIncludesTax()
                );
            }
        }

        $availablePoints = min(
            // Reduce the available points by the tax rate if they include, otherwise increase them
            $convertTax($customerPoints, $customerPointsIncludesTax),
            $pointsMax,
            $totalLimitPoints,
            $orderLimit ? $convertTax(
                $orderLimit->getTotalMax($pointsValue),
                $orderLimit->getIncludesTax()
            ) : $pointsValue
        );

        $minPoints = max(
            0,
            $pointsMin,
            $orderLimit ? $convertTax(
                $orderLimit->getTotalMin($pointsValue),
                $orderLimit->getIncludesTax()
            ) : 0
        );

        // Includes or excludes tax depending on incVat
        $requestedPoints = min($availablePoints, (int)$quote->getPointsWanted());

        // Zero it since we can't actually use any
        return $requestedPoints >= $minPoints ? $requestedPoints : 0;
    }

    /**
     * Collects itemized totals for rows and shipping.
     *
     * @param QuoteAddress $address
     */
    private function collectItems(
        Mage_Sales_Model_Quote_Address $address,
        QuoteAddressTotal $totals,
        bool $incVat,
        int $requestedPoints
    ): void {
        // RequestedPoints is always equal to or above minimum if not zero,
        // see getRequestedPoints.
        if( ! $requestedPoints) {
            // Zero means no points or below minimum, do not spread any points.
            // We have to exit since otherwise we will set items to their
            // minimum amount of points to spend.
            return;
        }

        /**
         * @var Quote $quote
         */
        $quote = $address->getQuote();
        $store = $quote->getStore();
        $items = $this->_getAddressItems($address);
        $getAmountTotal = $this->getAmountGetter($incVat);
        $pointsMin = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMin()));
        $pointsMax = (int)array_sum(array_map($getAmountTotal, $totals->getPointsMax()));

        $pointsNonMinMax = max($pointsMax - $pointsMin, 0);
        // Fraction of points between min and max to allocate
        $fracNonMinMax = $pointsNonMinMax > 0 ? max($requestedPoints - $pointsMin, 0) / $pointsNonMinMax : 0;

        /**
         * @var Array<string, Amount<float>>
         */
        $itemsRowPoints = [];
        /**
         * @var Array<string, Currency>
         */
        $itemsRowTotal = [];

        /**
         * @var QuoteAddressItem $i
         */
        foreach($items as $k => $i) {
            $rowValue = $totals->getRowPointsValue($i);

            if($rowValue) {
                $i->setPointsRowValue($rowValue->getTotalExclTax());
                $i->setPointsRowTaxValue($rowValue->getTax());

                $rowMax = $totals->getRowPointsMax($i) ?? $rowValue;
                $rowMin = $totals->getRowPointsMin($i) ?? new Amount(0, $incVat, 0);
                $rowCurrency = $totals->getRowCurrency($i) ?? new Currency(0, $incVat, 0, true);
                $rowMaxPoints = $getAmountTotal($rowMax);
                $rowMinPoints = $getAmountTotal($rowMin);

                $rowPoints = new Amount(
                    $rowMinPoints + $fracNonMinMax * max($rowMaxPoints - $rowMinPoints, 0),
                    $incVat,
                    $rowMin->getTax() + $fracNonMinMax * max($rowMax->getTax() - $rowMin->getTax(), 0)
                );

                $rowValuePoints = $getAmountTotal($rowValue);
                $rowFracPoints = $rowValuePoints > 0 ? $getAmountTotal($rowPoints) / $rowValuePoints : 0;

                $itemsRowPoints["idx_$k"] = $rowPoints;
                $itemsRowTotal["idx_$k"] = new Currency(
                    $rowFracPoints * $getAmountTotal($rowCurrency),
                    $incVat,
                    $rowFracPoints * $rowCurrency->getTax(),
                    true
                );
            }
        }

        $shippingValue = $totals->getShippingPointsValue();
        $shippingCurrency = $totals->getShippingCurrency() ?? new Currency(0, $incVat, 0, true);

        if($shippingValue) {
            $address->setPointsShippingValue($shippingValue->getTotalExclTax());
            $address->setPointsShippingTaxValue($shippingValue->getTax());

            $shippingValuePoints = $getAmountTotal($shippingValue);
            $shippingPoints = new Amount(
                // Shipping does not have min or max
                $fracNonMinMax * $shippingValuePoints,
                $incVat,
                $fracNonMinMax * $shippingValue->getTax()
            );

            $shippingFracPoints = $shippingValuePoints > 0 ? $getAmountTotal($shippingPoints) / $shippingValuePoints : 0;

            $itemsRowPoints["shipping"] = $shippingPoints;
            $itemsRowTotal["shipping"] = new Currency(
                $shippingFracPoints * $getAmountTotal($shippingCurrency),
                $incVat,
                $shippingFracPoints * $shippingCurrency->getTax(),
                true
            );
        }

        $itemsRowPointsSpread = spreadPoints($itemsRowPoints);
        $itemsRowTotalSpread = spreadCurrency($itemsRowTotal);

        /**
         * @var QuoteAddressItem $i
         */
        foreach($items as $k => $i) {
            if(array_key_exists("idx_$k", $itemsRowPointsSpread)) {
                $rowPoints = $itemsRowPointsSpread["idx_$k"];

                $i->setPointsRowPoints($rowPoints->getTotalExclTax());
                $i->setPointsRowTaxPoints($rowPoints->getTax());
            }

            if(array_key_exists("idx_$k", $itemsRowTotalSpread)) {
                $rowTotal = $itemsRowTotalSpread["idx_$k"];

                $i->setPointsRowTotal($rowTotal->getTotalExclTax());
                $i->setPointsRowTaxTotal($rowTotal->getTax());
                $i->setBasePointsRowTotal($store->convertPrice($rowTotal->getTotalExclTax(), false));
                $i->setBasePointsRowTaxTotal($store->convertPrice($rowTotal->getTax(), false));
            }
        }

        if(array_key_exists("shipping", $itemsRowPointsSpread)) {
            $shippingPoints = $itemsRowPointsSpread["shipping"];

            $address->setPointsShippingPoints($shippingPoints->getTotalExclTax());
            $address->setPointsShippingTaxPoints($shippingPoints->getTax());
        }

        if(array_key_exists("shipping", $itemsRowTotalSpread)) {
            $shippingTotal = $itemsRowTotalSpread["shipping"];

            $address->setPointsShippingTotal($shippingTotal->getTotalExclTax());
            $address->setPointsShippingTaxTotal($shippingTotal->getTax());
            $address->setBasePointsShippingTotal($store->convertPrice($shippingTotal->getTotalExclTax(), false));
            $address->setBasePointsShippingTaxTotal($store->convertPrice($shippingTotal->getTax(), false));
        }
    }

    /**
     * Collects discount total in points.
     *
     * @param QuoteAddress $address
     */
    private function collectDiscount(
        Mage_Sales_Model_Quote_Address $address,
        QuoteAddressTotal $totals,
        bool $incVat,
        float $fracPoints
    ): void {
        $quote = $address->getQuote();
        $store = $quote->getStore();

        $pointTaxRate = $this->getPointTaxRate($totals, $incVat);
        // Tax multiplier from a total including tax
        $taxMultiplier = $pointTaxRate / (1 + $pointTaxRate);

        $includedDiscount = array_filter($totals->getDiscount(), "Points\Core\Currency::isIncluded");
        $baseDiscountAmount = array_sum(array_map("Points\Core\Amount::totalInclTax", $includedDiscount));

        $pointsDiscount = (int)array_sum(array_map("Points\Core\Amount::totalExclTax", $totals->getPointsDiscount()));
        $pointsTaxDiscount = (int)array_sum(array_map("Points\Core\Amount::tax", $totals->getPointsDiscount()));

        $baseTotalAmount = $store->roundPrice($baseDiscountAmount * $fracPoints);
        $totalAmount = $store->convertPrice($baseTotalAmount, false);
        $pointDiscountParts = calculateRoundingSpread([
            "value" => $fracPoints * $pointsDiscount,
            "tax" => $fracPoints * $pointsTaxDiscount,
        ]);

        $baseAmount = spreadPrice([
            "value" => $baseTotalAmount * (1 - $taxMultiplier),
            "tax" => $baseTotalAmount * $taxMultiplier,
        ]);
        $amount = spreadPrice([
            "value" => $totalAmount * (1 - $taxMultiplier),
            "tax" => $totalAmount * $taxMultiplier,
        ]);

        $address->setPointsDiscountPoints($pointDiscountParts["value"]);
        $address->setPointsDiscountTaxPoints($pointDiscountParts["tax"]);
        $address->setPointsDiscountValue($pointsDiscount);
        $address->setPointsDiscountTaxValue($pointsTaxDiscount);
        $address->setPointsDiscountTotal($baseAmount["value"]);
        $address->setPointsDiscountTaxTotal($baseAmount["tax"]);
        $address->setBasePointsDiscountTotal($amount["value"]);
        $address->setBasePointsDiscountTaxTotal($amount["tax"]);
    }

    private function getPointTaxRate(
        QuoteAddressTotal $totals,
        bool $incVat
    ): float {
        $getAmountTotal = $this->getAmountGetter($incVat);

        $pointsValue = (int)array_sum(array_map($getAmountTotal, $totals->getPointsValue()));
        $pointsTax = array_sum(array_map("Points\Core\Amount::tax", $totals->getPointsValue()));

        // Rate from a total without tax
        $pointsValueExclTax = $incVat ? $pointsValue - $pointsTax : $pointsValue;

        return $pointsValueExclTax > 0 ? $pointsTax / $pointsValueExclTax : 0;
    }

    private function getAmountGetter(bool $incVat): callable {
        return $incVat ?
            "Points\Core\Amount::totalInclTax" :
            "Points\Core\Amount::totalExclTax";
    }
}
