<?php

declare(strict_types=1);

use Points\Core\QuoteAddressTotal;
use Points\Core\ProviderInterface;
use Points\Core\Extension\Quote as QuoteExt;
use Points\Core\Extension\QuoteAddress;

/**
 * Singleton caching the totals calculations.
 */
class Points_Core_Model_Cache_Quote {
    /**
     * @var SplObjectStorage<Mage_Sales_Model_Quote, array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}>
     */
    private $selected;

    /**
     * @var SplObjectStorage<Mage_Sales_Model_Quote, Array<array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}>>
     */
    private $available;

    public function __construct() {
        $this->available = new SplObjectStorage();
        $this->selected = new SplObjectStorage();
    }

    public function reset(Mage_Sales_Model_Quote $quote): void {
        unset($this->available[$quote]);
        unset($this->selected[$quote]);
    }

    /**
     * @return ?array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}
     */
    public function getSelectedTotal(Mage_Sales_Model_Quote $quote): ?array {
        if( ! $this->selected->contains($quote)) {
            $session = Mage::getSingleton("checkout/session");
            /**
             * @var QuoteExt $quote
             */
            $quote = $session->getQuote();
            $type = $quote->getPointsType();

            if( ! $type) {
                return null;
            }

            $helper = Mage::helper("points_core");
            $provider = $helper->getTypeProvider($type);

            if( ! $provider) {
                return null;
            }

            /**
             * @var QuoteAddress
             */
            $address = $quote->isVirtual() ?
                $quote->getBillingAddress() :
                $quote->getShippingAddress();

            $this->selected[$quote] = [
                "type" => $type,
                "provider" => $provider,
                "total" => QuoteAddressTotal::fromQuoteAddress($address, $type, $provider),
            ];
        }

        return $this->selected[$quote];
    }

    /**
     * @return Array<array{type:string, provider:ProviderInterface, total:QuoteAddressTotal}>
     */
    public function getAvailableTotals(Mage_Sales_Model_Quote $quote): array {
        if( ! $this->available->contains($quote)) {
            $helper = Mage::helper("points_core");
            $session = Mage::getSingleton("checkout/session");
            /**
             * @var QuoteExt $quote
             */
            $quote = $session->getQuote();
            /**
             * @var QuoteAddress
             */
            $address = $quote->isVirtual() ?
                $quote->getBillingAddress() :
                $quote->getShippingAddress();

            $data = [];
            $providers = $helper->getApplicableTypeProviders($quote);

            foreach($providers as $type => $provider) {
                $data[] = [
                    "type" => $type,
                    "provider" => $provider,
                    "total" => QuoteAddressTotal::fromQuoteAddress($address, $type, $provider),
                ];
            }

            $this->available[$quote] = $data;
        }

        return $this->available[$quote];
    }
}
