<?php

declare(strict_types=1);

use Points\Core\ProviderInterface;
use Points\Core\QuoteAddressTotal;
use Points\Core\Extension\Quote;

class Points_Core_Helper_Data extends Mage_Core_Helper_Data {
    const CONFIG_PROVIDERS = "points/providers";
    const SALES_POINTS_TYPE_KEY = "points_type";

    /**
     * Returns a list of *all* installed type providers, with the type
     * id as key.
     *
     * @return Array<string, ProviderInterface>
     */
    public function getTypeProviders(): array {
        /**
         * @var Array<string, array{model?: ?string}>
         */
        $data = Mage::getStoreConfig(self::CONFIG_PROVIDERS, Mage_Core_Model_Store::DEFAULT_CODE) ?: [];

        $items = array_combine(array_keys($data), array_map(function($item, $key) {
            $model = Mage::getSingleton($item["model"] ?? "");

            if( ! $model instanceof ProviderInterface) {
                throw new Exception(sprintf(
                    "%s: Model \"%s\" defined in %s/%s/model does not implement %s",
                    __CLASS__,
                    $item["model"] ?? "",
                    self::CONFIG_PROVIDERS,
                    $key,
                    ProviderInterface::class
                ));
            }

            return $model;
        }, $data, array_keys($data)));

        return $items;
    }

    /**
     * Returns a list of enabled type providers in the given store, with the type
     * id as key.
     *
     * @return Array<string, ProviderInterface>
     */
    public function getEnabledTypeProviders(Mage_Core_Model_Store $store): array {
        return array_filter(
            $this->getTypeProviders(),
            function($p) use($store): bool {
                return $p->isEnabled($store);
            }
        );
    }

    /**
     * Returns a list of type providers which apply to the quote, with the type
     * id as key.
     *
     * @return Array<string, ProviderInterface>
     */
    public function getApplicableTypeProviders(Mage_Sales_Model_Quote $quote): array {
        return array_filter(
            $this->getTypeProviders(),
            function($p) use($quote): bool {
                return $p->appliesTo($quote);
            }
        );
    }

    public function getTypeProvider(
        string $type
    ): ?ProviderInterface {
        /**
         * @var Array<string, array{model?: ?string}>
         */
        $data = Mage::getStoreConfig(self::CONFIG_PROVIDERS, Mage_Core_Model_Store::DEFAULT_CODE) ?: [];

        if(empty($data[$type]["model"])) {
            return null;
        }

        $model = Mage::getSingleton($data[$type]["model"]);

        if( ! $model instanceof ProviderInterface) {
            throw new Exception(sprintf(
                "%s: Model \"%s\" defined in %s/%s/model does not implement %s",
                __METHOD__,
                $data[$type]["model"] ?? "",
                self::CONFIG_PROVIDERS,
                $type,
                ProviderInterface::class
            ));
        }

        return $model;
    }

    public function getAdminhtmlProductEditTab(): array {
        return [
            "label" => $this->__("Points Payment"),
            "title" => $this->__("Points Payment"),
            "url" => Mage::app()->getStore()->getUrl("*/catalog_product_points", ["_current" => true]),
            "class" => "ajax",
        ];
    }

    /**
     * @param Quote $quote
     */
    public function resetQuotePoints(Mage_Sales_Model_Quote $quote): void {
        $quote->setPointsValue(0);
        $quote->setPointsTaxValue(0);
        $quote->setPointsPoints(0);
        $quote->setPointsTaxPoints(0);
        $quote->setPointsAmount(0.0);
        $quote->setPointsTaxAmount(0.0);
        $quote->setBasePointsAmount(0.0);
        $quote->setBasePointsTaxAmount(0.0);

        $quote->setPointsAmountIncluded(0.0);
        $quote->setPointsTaxAmountIncluded(0.0);
        $quote->setPointsAmountExcluded(0.0);
        $quote->setPointsTaxAmountExcluded(0.0);

        $quote->setPointsShippingValue(null);
        $quote->setPointsShippingTaxValue(null);
        $quote->setPointsShippingPoints(0);
        $quote->setPointsShippingTaxPoints(0);
        $quote->setPointsShippingTotal(0.0);
        $quote->setPointsShippingTaxTotal(0.0);
        $quote->setBasePointsShippingTotal(0.0);
        $quote->setBasePointsShippingTaxTotal(0.0);

        $quote->setPointsDiscountValue(0);
        $quote->setPointsDiscountTaxValue(0);
        $quote->setPointsDiscountPoints(0);
        $quote->setPointsDiscountTaxPoints(0);
        $quote->setPointsDiscountTotal(0.0);
        $quote->setPointsDiscountTaxTotal(0.0);
        $quote->setBasePointsDiscountTotal(0.0);
        $quote->setBasePointsDiscountTaxTotal(0.0);

        $quoteCache = Mage::getSingleton("points_core/cache_quote");

        $quoteCache->reset($quote);
    }
}
