<?php

/**
 * @method getStore
 */
class Crossroads_PayEx_Model_Api extends Mage_Core_Model_Abstract {
    /**
     * @param  Mage_Sales_Model_Quote
     * @return Array<string> Order reference, redirect url
     */
    public function initialize8($quote) {
        $method = $quote->getPayment()->getMethodInstance();
        $store  = $quote->getStore();

        if( ! $method || ! $method instanceof Crossroads_PayEx_Model_Method_Abstract) {
            throw new Exception(sprintf(
                "%s: Method instance '%s' is not an instance of Crossroads_PayEx_Model_Method_Abstract",
                is_object($method) ? get_class($method) : gettype($method)));
        }

        $params = [
            "accountNumber"     => $method->getConfigAccountNumber(),
            "purchaseOperation" => $method->getPurchaseOperation(),
            "price"             => round(100 * $quote->getGrandTotal()),
            "priceArgList"      => "",
            "currency"          => $quote->getQuoteCurrencyCode(),
            "vat"               => 0, // ???: From PayEx module
            "orderID"           => $quote->getReservedOrderId(),
            "productNumber"     => $quote->getReservedOrderId(),
            "description"       => $store->getName(),
            "clientIPAddress"   => Mage::helper("core/http")->getRemoteAddr(),
            "clientIdentifier"  => "USERAGENT=".rawurlencode(Mage::helper("core/http")->getHttpUserAgent()),
            "additionalValues"  => http_build_query($method->getPayExInitializeAdditionalData($quote)),
            "externalID"        => "",
            "returnUrl"         => $store->getUrl(null, [
                "_direct"       => $method->getConfigReturnUrl(),
                "_use_rewrite"  => false,
                "_store_to_url" => true,
            ]),
            "view"              => $method->getPayExViewParameter(),
            "agreementRef"      => "",
            "cancelUrl"         => $store->getUrl(null, [
                "_direct"       => $method->getConfigCancelUrl(),
                "_use_rewrite"  => false,
                "_store_to_url" => true,
            ]),
            "clientLanguage"    => $method->getConfigClientLanguage(),
        ];

        $params["hash"] = $method->calculateHash($params);

        $client = new SoapClient($method->getPayExOrderWsdl());
        $result = new SimpleXMLElement($client->Initialize8($params)->Initialize8Result);

        Mage::log(sprintf("%s: result:%s", __METHOD__, $result->asXML()));

        if($result->status && $result->status->code && $result->status->code == "OK") {
            return [
                (string)$result->orderRef,
                (string)$result->redirectUrl,
            ];
        }

        // Log params too, if something went wrong
        Mage::log(sprintf("%s: params:%s", __METHOD__, json_encode($params)));

        $e = new Exception(sprintf(
            "%s: PayEx call failed with %s",
            __METHOD__,
            $result->status ? $result->status->code : "Unknown"
        ));

        $e->payexCode = $result->status ? $result->status->code : "Unknown";

        throw $e;
    }

    /**
     * @param  Mage_Payment_Model_Info
     */
    public function getApprovedDeliveryAddress($paymentInfo, $orderRef) {
        $method = $paymentInfo->getMethodInstance();

        if( ! $method || ! $method instanceof Crossroads_PayEx_Model_Method_Abstract) {
            throw new Exception(sprintf(
                "%s: Method instance '%s' is not an instance of Crossroads_PayEx_Model_Method_Abstract",
                is_object($method) ? get_class($method) : gettype($method)));
        }

        $params = [
            "accountNumber" => $method->getConfigAccountNumber(),
            "orderRef"      => $orderRef,
        ];

        $params["hash"] = $method->calculateHash($params);

        $client = new SoapClient($method->getPayExOrderWsdl());
        $result = new SimpleXMLElement($client->GetApprovedDeliveryAddress($params)->GetApprovedDeliveryAddressResult);

        Mage::log(sprintf("%s: result:%s", __METHOD__, $result->asXML()));

        if($result->status && $result->status->code && $result->status->errorCode == "OK") {
            return [
                "prefix"     => null,
                "firstname"  => (string)$result->firstName,
                "middlename" => null,
                "lastname"   => (string)$result->lastName,
                "suffix"     => null,
                "company"    => null,
                "street"     => implode("\n", array_values(array_filter(array_map("trim", [
                    (string)$result->address1,
                    (string)$result->address2,
                    (string)$result->address3,
                ])))),
                "city"       => (string)$result->city,
                "region_id"  => null,
                "postcode"   => (string)$result->postalCode,
                "country_id" => (string)$result->country,
                // Telephone can be in format SE+46-71234567
                "telephone"  => strpos((string)$result->phone, "+") !== false ?
                    strstr((string)$result->phone, "+") : (string)$result->phone,
                "email"      => (string)$result->eMail,
            ];
        }

        // Log params too, if something went wrong
        Mage::log(sprintf("%s: params:%s", __METHOD__, json_encode($params)));

        $e = new Exception(sprintf(
            "%s: PayEx call failed with %s",
            __METHOD__,
            $result->status ? $result->status->code : "Unknown"
        ));

        $e->payexCode = $result->status ? $result->status->code : "Unknown";

        throw $e;
    }

    /**
     * @param  Mage_Payment_Model_Info
     */
    public function finalizeTransaction($paymentInfo, $orderRef, $amount, $vatAmount) {
        $method = $paymentInfo->getMethodInstance();

        if( ! $method || ! $method instanceof Crossroads_PayEx_Model_Method_Abstract) {
            throw new Exception(sprintf(
                "%s: Method instance '%s' is not an instance of Crossroads_PayEx_Model_Method_Abstract",
                is_object($method) ? get_class($method) : gettype($method)));
        }

        $params = [
            "accountNumber"   => $method->getConfigAccountNumber(),
            "orderRef"        => $orderRef,
            "amount"          => round(100 * $amount),
            "vatAmount"       => 0, // ???: From PayEx module
            "clientIPAddress" => Mage::helper("core/http")->getRemoteAddr(),
        ];

        $params["hash"] = $method->calculateHash($params);

        $client = new SoapClient($method->getPayExOrderWsdl());
        $result = new SimpleXMLElement($client->FinalizeTransaction($params)->FinalizeTransactionResult);

        Mage::log(sprintf("%s: result:%s", __METHOD__, $result->asXML()));

        if($result->status && $result->status->code && $result->status->errorCode == "OK") {
            return [
                "transactionStatus" => (int)(string)$result->transactionStatus,
                "transactionNumber" => (string)$result->transactionNumber,
                "orderRef"          => (string)$result->orderRef,
                "paymentMethod"     => (string)$result->paymentMethod,
                "raw"               => $result->asXML(),
            ];
        }

        // Log params too, if something went wrong
        Mage::log(sprintf("%s: params:%s", __METHOD__, json_encode($params)));

        $e = new Exception(sprintf(
            "%s: PayEx call failed with %s",
            __METHOD__,
            $result->status ? $result->status->code : "Unknown"
        ));

        $e->payexCode = $result->status ? $result->status->code : "Unknown";

        throw $e;
    }

    /**
     * @param  Crossroads_PayEx_Model_Method_Abstract
     */
    protected function getSoapClient($method) {
        $client = new SoapClient($method->getPayExOrderWsdl());
    }
}
