<?php

declare(strict_types=1);

use GraphQL\Error\Warning;
use GraphQL\Type\Schema;

use MageQL\Registry;
use MageQL\Context;
use MageQL\Schema\SchemaInterface;
use MageQL\Schema\DefaultSchema;

class MageQL_Core_Helper_Data extends Mage_Core_Helper_Abstract {
    /** @const string */
    const LOG_CHANNEL = 'mageql.log';
    /**
     * Boolean configuration value to enable GraphQL Login.
     */
    const CONFIG_CUSTOMER_STARTUP_ENABLE_LOGIN = "customer/startup/graphql_enable_login";
    /**
     * Boolean configuration value to enable GraphQL introspection
     */
    const CONFIG_QUERY_ENABLE_INTROSPECTION = "mageql/query/enable_introspection";
    /**
     * When this key is provided you could bypass a disabled GraphQL introspection
     */
    const CONFIG_QUERY_INTROSPECTION_KEY = "mageql/query/introspection_key";
    /**
     * Boolean configuration value to enable GraphQL suggestions
     */
    const CONFIG_QUERY_ENABLE_SUGGEST = "mageql/query/enable_suggest";
    /**
     * Int configuration value for max query depth
     */
    const CONFIG_QUERY_DEPTH = "mageql/query/depth";
    /**
     * Int configuration value for max query complexity
     */
    const CONFIG_QUERY_MAX_COMPLEXITY = "mageql/query/max_complexity";
    /**
     * Int configuration value for warn query complexity
     */
    const CONFIG_QUERY_WARN_COMPLEXITY = "mageql/query/warn_complexity";

    /**
     * Creates a dynamically expanding schema instance based on the context.
     *
     * Defaults:
     *
     *  * unreachable: false
     *  * federation: true
     *
     * @param array{ unreachable?: bool, federation?: bool } $include
     */
    public function loadSchema(
        Context $context,
        array $include = []
    ): Schema {
        $schemaName = $context->getSchemaName();
        $config = $context->getConfig(sprintf("mageql/schema/%s", $schemaName));
        $factories = [];

        if( ! $config) {
            throw new MageQL_Core_Exception_SchemaNotFound($schemaName);
        }

        $factories[] = new DefaultSchema();

        // TODO: Do we reverse this so we can have priority?
        foreach($config as $k => $v) {
            if(array_key_exists("model", $v)) {
                /**
                 * @var false | SchemaInterface
                 */
                $model = Mage::getSingleton($v["model"]);

                if( ! $model instanceof SchemaInterface) {
                    throw new Exception(sprintf(
                        "%s: Model instantiated from config '%s' must be an instance of %s",
                        __METHOD__,
                        sprintf("config/default/mageql/schema/%s/%s/model", $schemaName, $k),
                        SchemaInterface::class
                    ));
                }

                $factories[] = $model;

                continue;
            }

            throw new Exception(sprintf("%s: Missing model tag in 'mageql/schema/%s/%s'.", __METHOD__, $schemaName, $k));
        }

        if($include["federation"] ?? true) {
            $factories[] = new MageQL_Core_Model_Schema_Federation();
        }

        /**
         * In case we get warnings we need to keep track of them
         *
         * @var Array<array{id:string, msg:string}>
         */
        $warnings = [];

        if(Mage::getIsDeveloperMode()) {
            Warning::setWarningHandler(function(string $msg, int $id) use(&$warnings) {
                $warnings[] = [
                    "id"  => $id,
                    "msg" => $msg,
                ];
            });
        }

        $registry = new Registry($factories, [
            "includeUnreachable" => $include["unreachable"] ?? false,
        ]);

        $schema = $registry->getSchema();

        if( ! empty($warnings)) {
            // Assert the schema immediately, probably why we got a warning
            $schema->assertValid();

            throw new Exception(sprintf(
                "%s: Warnings: %s",
                __METHOD__,
                implode(", ", array_map(function(array $warning) {
                    return sprintf("%s: %s", $warning["id"], $warning["msg"]);
                }, $warnings))
            ));
        }

        if(Mage::getIsDeveloperMode()) {
            Warning::setWarningHandler(null);
        }

        return $schema;
    }
}
