<?php

declare(strict_types=1);

use GraphQL\Error\ClientAware;
use GraphQL\Error\DebugFlag;
use GraphQL\Error\Error;
use GraphQL\Error\FormattedError;
use GraphQL\Executor\ExecutionResult;
use GraphQL\Server\Helper;
use GraphQL\Server\OperationParams;
use GraphQL\Server\RequestError;
use GraphQL\Server\StandardServer;
use GraphQL\Utils\Utils;
use GraphQL\Validator\DocumentValidator;
use GraphQL\Validator\Rules;
use MageQL\Context;
use MageQL\InactiveStoreException;
use MageQL\NotInstalledException;
use MageQL\SessionFailedException;

use function MageQL\handleGraphQLError;

class MageQL_Core_Router_GraphQL extends Mage_Core_Controller_Varien_Router_Abstract {
    /**
     * Default schema name.
     */
    const SCHEMA_DEFAULT = "default";
    /**
     * Design area we run GraphQL in.
     */
    const GRAPHQL_AREA = "frontend";
    /**
     * Path segment for graphql API calls.
     */
    const GRAPHQL_PATH = "graphql";
    /**
     * Path segment for internal graphql API calls.
     */
    const GRAPHQL_INTERNAL_PATH = "graphql-internal";
    /**
     * Path segment for interactive graphql explorer.
     */
    const GRAPHIQL_PATH = "graphiql";

    /**
     * Event triggered after the context and session has been initialized but
     * before the Schema has been initialized or the the request body has
     * started processing.
     *
     * Parameters:
     *
     *  * context: MageQL\Context
     */
    const EVENT_CONTEXT_AFTER = "mageql_core_context_after";

    /**
     * @param string $unusedArea
     * @param bool $unusedUseNames
     * @return void
     */
    public function collectRoutes($unusedArea, $unusedUseNames) {
        // Intentionally left empty
    }

    /**
     * Session namespace to refer in other places
     */
    public function getSessionNamespace(): string {
        $namespace = trim(strtolower(Mage::getStoreConfig("web/cookie/frontend_namespace")));

        if(empty($namespace)) {
            $namespace = Mage_Core_Controller_Front_Action::DEFAULT_SESSION_NAMESPACE;
        }

        if($namespace == Mage_Adminhtml_Controller_Action::SESSION_NAMESPACE) {
            throw new RuntimeException(sprintf(
                "%s: Session namespace matches admin namespace '%s'.",
                __METHOD__,
                Mage_Adminhtml_Controller_Action::SESSION_NAMESPACE
            ));
        }

        return $namespace;
    }

    public function match(Zend_Controller_Request_Http $request): bool {
        $path   = array_filter(explode("/", trim($request->getPathInfo(), "/")));
        $front  = $this->getFront();
        $type   = $path[0] ?? null;
        $schema = $path[1] ?? null;

        switch($type) {
        case self::GRAPHIQL_PATH:
            $request->setDispatched(true);
            $this->graphiQL($front->getResponse(), $schema ?: self::SCHEMA_DEFAULT);

            return true;

        case self::GRAPHQL_PATH:
            $request->setDispatched(true);
            $this->graphQL($request, $front->getResponse(), $schema ?: self::SCHEMA_DEFAULT, true);

            return true;

        case self::GRAPHQL_INTERNAL_PATH:
            $request->setDispatched(true);
            $this->graphQL($request, $front->getResponse(), $schema ?: self::SCHEMA_DEFAULT, false);

            return true;

        default:
            return false;
        }
    }

    public function graphiQL(
        Zend_Controller_Response_Abstract $response,
        string $schemaName
    ): void {
        /**
         * @var MageQL_Core_Block_Graphiql $block
         */
        $block = Mage::getSingleton("core/layout")
            ->createBlock("mageql/graphiql");

        if($schemaName !== self::SCHEMA_DEFAULT) {
            $block->setSchemaName($schemaName);
        }

        $response->setBody($block->toHtml());
    }

    /**
     * Writes the supplied `$result` to `$response` and marks `$request` as
     * dispatched, `$code` will be set as the response code if it is larger
     * than the already set code on `$response`.
     *
     * @param mixed $result
     */
    public function setJsonResponse(
        Zend_Controller_Response_Abstract $response,
        $result,
        int $code = 200
    ): void {
        // Only overwrite the response code if we have not already set one,
        // or if we get an exception/user-error
        if($response->getHttpResponseCode() < $code) {
            $response->setHttpResponseCode($code);
        }

        $jsonFlags = JSON_PRESERVE_ZERO_FRACTION // Preserve the floats
            | JSON_UNESCAPED_SLASHES // Escaped slashes is an annoyance and a non-standard part of JSON
            | JSON_UNESCAPED_UNICODE // We send full unicode encoded anyway
            | (Mage::getIsDeveloperMode() ? JSON_PRETTY_PRINT : 0); // Make it easy to read

        $response->setHeader("Content-Type", "application/json; charset=utf-8", true)
                 ->setBody(json_encode($result, $jsonFlags));
    }

    public function graphQL(
        Zend_Controller_Request_Http $request,
        Zend_Controller_Response_Abstract $response,
        string $schemaName,
        bool $external = false
    ): void {
        $debug = Mage::getIsDeveloperMode() ?
            DebugFlag::INCLUDE_DEBUG_MESSAGE | DebugFlag::INCLUDE_TRACE :
            DebugFlag::NONE;

        /**
         * Suppress the error here since we are certain to not implement the
         * interface on anything not throwable.
         *
         * @psalm-suppress InvalidCatch
         */
        try {
            $app = Mage::app();
            $store = $app->getStore();
            $helper = Mage::helper("mageql/data");
            $session = $this->initSession($app, $store);

            // TODO: Wrap cart, session and store in one?
            $context = new MageQL_Core_Model_Context($store, $session, $schemaName);

            $customValidation = [];
            if ($external) {
                $customValidation = $this->customValidation($context, $request);
            }
            $validationRules = array_merge(
                DocumentValidator::defaultRules(),
                $customValidation
            );

            Mage::dispatchEvent(self::EVENT_CONTEXT_AFTER, [
                "context" => $context,
            ]);

            // Always include unreachable if we are in debug or if we are querying from GraphiQL
            $includeUnreachable = Mage::getIsDeveloperMode() ||
                strpos($request->getHeader("Referer") ?: "", self::GRAPHIQL_PATH) !== false;
            $schema = $helper->loadSchema($context, [
                "unreachable" => $includeUnreachable,
            ]);
            $server = new StandardServer([
                "schema" => $schema,
                "rootValue" => $store,
                "context" => $context,
                "queryBatching" => true,
                "validationRules" => array_values($validationRules),
                "errorsHandler" => [$this, "handleError"],
                "debugFlag" => $debug,
            ]);

            $data = $this->parseHttpRequest($request, $server->getHelper());

            /**
             * We do not use promises, force-cast it to something without
             * promises.
             *
             * @var ExecutionResult|Array<ExecutionResult>
             */
            $result = $server->executeRequest($data);

            $responseCode = $this->getResponseCode($result);
            $respData = is_array($result) ? array_map(function($r) use($debug) {
                return $r->toArray($debug);
            }, $result) : $result->toArray($debug);

            $this->setJsonResponse($response, $respData, $responseCode);
        }
        // RequestError is ClientAware
        catch(ClientAware $e) {
            if($e->isClientSafe()) {
                Mage::logException($e, "client_errors", Zend_Log::DEBUG);
            }

            $this->setJsonResponse($response, [
                "errors" => [
                    handleGraphQLError($e, function(Throwable $e) use($debug): array {
                        return FormattedError::createFromException($e, $debug);
                    }),
                ],
            ], 400);
        }
    }

    public function customValidation(
        Context $context,
        Zend_Controller_Request_Http $request
    ): array {
        $maxComplexity = $context->getConfig(MageQL_Core_Helper_Data::CONFIG_QUERY_MAX_COMPLEXITY) ?? 0;
        $warnComplexity = $context->getConfig(MageQL_Core_Helper_Data::CONFIG_QUERY_WARN_COMPLEXITY) ?? 0;
        $queryDepth = $context->getConfig(MageQL_Core_Helper_Data::CONFIG_QUERY_DEPTH) ?? 0;

        $customValidation = [
            Rules\QueryComplexity::class => new MageQL_Core_Model_Validation_QueryComplexity($maxComplexity, $warnComplexity),
            Rules\QueryDepth::class => new Rules\QueryDepth($queryDepth),
        ];

        // Disable 'Did you mean...'
        $enableSuggest = $context->getConfig(MageQL_Core_Helper_Data::CONFIG_QUERY_ENABLE_SUGGEST) ?? 1;
        if (!$enableSuggest) {
            $customValidation[Rules\FieldsOnCorrectType::class] = new MageQL_Core_Model_Validation_FieldsOnCorrectType();
            $customValidation[Rules\KnownArgumentNames::class] = new MageQL_Core_Model_Validation_KnownArgumentNames();
            $customValidation[Rules\KnownTypeNames::class] = new MageQL_Core_Model_Validation_KnownTypeNames();
        }

        $introspectionConfig = $context->getConfig(MageQL_Core_Helper_Data::CONFIG_QUERY_ENABLE_INTROSPECTION) ?? 1;
        $introspectionConfigKey = $context->getConfig(MageQL_Core_Helper_Data::CONFIG_QUERY_INTROSPECTION_KEY) ?? "";
        $providedIntrospectionKey = $request->getHeader("X-Introspection-Key") 
            ? $request->getHeader("X-Introspection-Key")
            : $request->getParam("introspectionKey", "");
        $enableIntrospection = $introspectionConfig ?: ($introspectionConfigKey === $providedIntrospectionKey);
        if (!$enableIntrospection) {
            $customValidation[Rules\DisableIntrospection::class] = new Rules\DisableIntrospection(Rules\DisableIntrospection::ENABLED);
        }

        return $customValidation;
    }

    public function initSession(
        Mage_Core_Model_App $app,
        Mage_Core_Model_Store $store
    ): Mage_Core_Model_Session_Abstract_Varien {
        // Adapted from Mage_Core_Controller_Varien_Action::preDispatch()
        if( ! Mage::isInstalled()) {
            throw new NotInstalledException();
        }

        if( ! $store->getIsActive()) {
            throw new InactiveStoreException();
        }

        $layout = Mage::getSingleton("core/layout");
        $namespace = $this->getSessionNamespace();

        $layout->setArea(self::GRAPHQL_AREA);

        // Boot session, attempt to restart it if it is invalid.
        //
        // Alternate version of the session-start code in
        // Mage_Core_Controller_Varien_Action::preDispatch(), which had the issue
        // of failing to restart the session properly and instead causing errors
        // to become redirects which is incompatible with an API.
        try {
            $session = Mage::getSingleton("core/session", ["name" => $namespace])
                ->start();
        }
        catch(Mage_Core_Model_Session_Exception $_e) {
            if(PHP_SAPI !== "cli") {
                // The session is invalid, destroy if it is active
                if(session_status() == PHP_SESSION_ACTIVE) {
                    session_destroy();
                    session_commit();
                }

                // New id to reinitialize
                if( ! session_regenerate_id(true)) {
                    Mage::log(sprintf(
                        "%s: Failed to regenerate session id",
                        __METHOD__
                    ));
                }
            }

            // Try to reinitialize session
            try {
                // TODO: Do we try to reset the magento instance of the session?
                $session = Mage::getSingleton("core/session", ["name" => $namespace])
                    ->start()
                    // Core is from Mage_Core_Model_Session::__construct()
                    // init is called to make sure the session is properly initialized
                    ->init("core", $namespace);
            }
            catch(Exception $e) {
                throw new SessionFailedException($e);
            }
        }

        // Code from Mage_Core_Controller_Varien_Action::preDispatch() again
        $app->loadArea($layout->getArea());

        return $session;
    }

    public function handleError(array $errors, callable $formatter): array {
        return array_map(function(Throwable $error) use($formatter): array {
            if($error instanceof ClientAware) {
                if($error->isClientSafe()) {
                    Mage::logException($error, "client_errors", Zend_Log::DEBUG);
                }
                else {
                    $prev = $error->getPrevious() ?: $error;

                    if( ! $prev instanceof Exception) {
                        Mage::logException(new Exception($prev->getMessage(), (int)$prev->getCode(), $prev));
                    }
                    else {
                        Mage::logException($prev);
                    }
                }
            }
            else {
                Mage::logException($error);
            }

            return handleGraphQLError($error, $formatter);
        }, $errors);
    }

    /**
     * @param (ExecutionResult|Array<ExecutionResult>) $result
     */
    public function getResponseCode($result): int {
        $error = 400;

        foreach(is_array($result) ? $result : [$result] as $r) {
            /**
             * This can actually be null, the constructor even defaults it to null.
             *
             * @var mixed[]|null
             */
            $data = $r->data;

            if($data !== null || empty($r->errors)) {
                return 200;
            }
            else if( ! empty($r->errors)) {
                foreach($r->errors as $e) {
                    if($e->getCategory() === Error::CATEGORY_INTERNAL) {
                        $error = 500;
                    }
                }
            }
        }

        return $error;
    }

    /**
     * Implmentation of the GraphQL\Server\Helper::parseHttpRequest() for Magento.
     *
     * @return OperationParams|Array<OperationParams>
     */
    public function parseHttpRequest(Zend_Controller_Request_Http $request, Helper $helper) {
        $method = $request->getMethod();
        $bodyParams = [];
        $urlParams = $request->getQuery();

        if($method === "POST") {
            $contentType = $request->getHeader("Content-Type");

            if( ! $contentType) {
                throw new RequestError('Missing "Content-Type" header');
            }

            if(stripos($contentType, "application/graphql") !== false) {
                $bodyParams = ['query' => $request->getRawBody() ?: ''];
            }
            elseif(stripos($contentType, "application/json") !== false) {
                $bodyParams = json_decode($request->getRawBody() ?: "", true);

                if(json_last_error()) {
                    throw new RequestError("Could not parse JSON: " . json_last_error_msg());
                }

                if( ! is_array($bodyParams)) {
                    throw new RequestError(
                        "GraphQL Server expects JSON object or array, but got " .
                        Utils::printSafeJson($bodyParams)
                    );
                }
            }
            // We do not allow "application/x-www-form-urlencoded" anymore due
            // to Cross-Origin Resource Sharing (CORS) allowing POST requests
            // using this content type as "simple requests", bypassing any CORS
            // policies.
            else {
                throw new RequestError("Unexpected content type: " . Utils::printSafeJson($contentType));
            }
        }

        return $helper->parseRequestParams($method, $bodyParams, $urlParams);
    }
}
