<?php

declare(strict_types=1);

use MageQL\Type\InputFieldBuilder;

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class MageQL_Core_Model_Form_Customer_Address extends MageQL_Core_Model_Form_Abstract {
    public function getCustomFields(): array {
        return [
            "street" => new InputFieldBuilder(
                "[String!]", "Street, one line per array item"
            ),
            "regionCode" => new InputFieldBuilder(
                "ID", "Region code"
            ),
            "countryCode" => new InputFieldBuilder(
                "ID", "Country code"
            ),
        ];
    }

    public function includeAttribute(
        Mage_Eav_Model_Attribute $attribute
    ): bool {
        return !in_array($attribute->getAttributeCode(), [
            "region",
            "country_id",
            "region_id",
            "street",
        ]);
    }

    public function getForm(): Mage_Eav_Model_Form {
        $addressResource = Mage::getResourceModel("customer/address");
        $addressForm = Mage::getModel("customer/form");

        $addressForm->setFormCode("customer_address_edit");
        $addressForm->setEntityType((int)$addressResource->getEntityType()->getId());

        return $addressForm;
    }

    public function setFields(Varien_Object $obj, array $data): bool {
        $helper = Mage::helper("mageql/address");
        $modified = parent::setFields($obj, $data);

        if(array_key_exists("street", $data)) {
            $newStreet = trim(implode("\n", $data["street"]));
            $oldStreet = trim(implode("\n", (array)$obj->getStreet()));

            if($newStreet !== $oldStreet) {
                // Since we have already trimmed it and imploded it we
                // do no have to call implodeStreetAddress
                $obj->setStreet($newStreet);

                $modified = true;
            }
        }

        // Country before region, to be able to validate
        if(array_key_exists("countryCode", $data) && $data["countryCode"] !== $obj->getCountryId()) {
            $approvedIds = $helper->getAllowedCountryCodes(Mage::app()->getStore());
            $countryId = strtoupper(trim($data["countryCode"]));

            if( ! in_array($countryId, $approvedIds)) {
                // We throw since the frontend should only allow the pre-approved countries
                throw new MageQL_Core_InvalidCountryCodeException($countryId);
            }

            $obj->setCountryId($countryId);

            $modified = true;
        }

        if(array_key_exists("regionCode", $data) && $data["regionCode"] !== $obj->getRegionCode()) {
            // TODO: Validate
            $obj->setRegionCode($data["regionCode"]);
            // Reset
            $obj->unsRegionId();
            $obj->unsRegion();

            $modified = true;
        }

        return $modified;
    }

    public function copyData(
        Varien_Object $from,
        Varien_Object $to,
        bool $skipNull = false
    ): bool {
        $modified = parent::copyData($from, $to, $skipNull);

        foreach(["region_id", "country_id", "street"] as $field) {
            if($this->copyDataField($from, $to, $field, $skipNull)) {
                $modified = true;
            }
        }

        return $modified;
    }
}
