<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

/**
 * @psalm-type AddressInput array{
 *   prefix?: string,
 *   firstname?: string,
 *   middlename?: string,
 *   lastname?: string,
 *   suffix?: string,
 *   company?: string,
 *   street?: Array<string>,
 *   city?: string,
 *   postcode?: string,
 *   regionCode?: string,
 *   countryCode?: string,
 *   telephone?: string
 * }
 */
class MageQL_Core_Model_Customer_Address {
    /**
     * Event triggered when a customer address was successfully created by the
     * createCustomerAddress mutation.
     *
     * Parameters:
     *
     *  * address: MageQL_Customer_Model_Address
     *  * context: MageQL_Core_Model_Context
     *  * customer: Mage_Customer_Model_Customer
     */
    const EVENT_MUTATION_CREATE_CUSTOMER_ADDRESS_SUCCESS = "mageql_core_mutation_createCustomerAddress_success";

    /**
     * Event triggered when a customer address was successfully updated by the
     * updateCustomerAddress mutation.
     *
     * Parameters:
     *
     *  * address: MageQL_Customer_Model_Address
     *  * context: MageQL_Core_Model_Context
     *  * customer: Mage_Customer_Model_Customer
     */
    const EVENT_MUTATION_UPDATE_CUSTOMER_ADDRESS_SUCCESS = "mageql_core_mutation_updateCustomerAddress_success";

    public static function resolveIsDefaultBilling(
        Mage_Customer_Model_Address $src
    ): bool {
        $customer = $src->getCustomer();

        if( ! $customer) {
            throw new Exception(sprintf("%s: Address was not loaded through customer", __METHOD__));
        }

        return $src->getId() === $customer->getData("default_billing");
    }

    public static function resolveIsDefaultShipping(
        Mage_Customer_Model_Address $src
    ): bool {
        $customer = $src->getCustomer();

        if( ! $customer) {
            throw new Exception(sprintf("%s: Address was not loaded through customer", __METHOD__));
        }

        return $src->getId() === $customer->getData("default_shipping");
    }

    /**
     * @param mixed $unusedSrc
     * @param array{
     *   address:AddressInput,
     *   setDefaultBilling: bool,
     *   setDefaultShipping: bool
     * } $args
     */
    public static function mutateCreate(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Create {
        $session = Mage::getSingleton("customer/session");
        $form = Mage::getSingleton("mageql/form_customer_address");
        $customer = $session->isLoggedIn() ? $session->getCustomer() : null;

        if( ! $customer) {
            return new MageQL_Core_Model_Customer_Result_Address_Create(
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_NOT_LOGGED_IN
            );
        }

        $address = Mage::getModel("customer/address");
        $modified = $form->setFields($address, $args["address"]);

        if( ! $modified) {
            return new MageQL_Core_Model_Customer_Result_Address_Create(
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_INVALID_ADDRESS,
                $customer,
                $address
            );
        }

        $address->setCustomerId((int)$customer->getId());

        $result = $address->validate();

        if($result !== true) {
            return new MageQL_Core_Model_Customer_Result_Address_Create(
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_INVALID_ADDRESS,
                $customer,
                $address
            );
        }

        $address->save();
        // Clear all addresses since to force a reload of the address collection
        // if necessary (this is not a delete).
        $customer->cleanAllAddresses();

        if($args["setDefaultBilling"]) {
            $customer->setDefaultBilling((int)$address->getId());
        }

        if($args["setDefaultShipping"]) {
            $customer->setDefaultShipping((int)$address->getId());
        }

        $customer->save();

        Mage::dispatchEvent(self::EVENT_MUTATION_CREATE_CUSTOMER_ADDRESS_SUCCESS, [
            "address" => $address,
            "context" => $ctx,
            "customer" => $customer,
        ]);

        return new MageQL_Core_Model_Customer_Result_Address_Create(
            MageQL_Core_Model_Customer_Result_Address_Create::SUCCESS,
            $customer,
            $address
        );
    }

    /**
     * @param mixed $unusedSrc
     * @param array{id:string, address:AddressInput} $args
     */
    public static function mutateUpdate(
        $unusedSrc,
        array $args,
        MageQL_Core_Model_Context $ctx
    ): MageQL_Core_Model_Customer_Result_Address_Update {
        $session = Mage::getSingleton("customer/session");
        $addressId = $args["id"];
        $customer = $session->isLoggedIn() ? $session->getCustomer() : null;
        $form = Mage::getSingleton("mageql/form_customer_address");

        if( ! $customer) {
            return new MageQL_Core_Model_Customer_Result_Address_Update(
                MageQL_Core_Model_Customer_Result_Address_Update::ERROR_NOT_LOGGED_IN
            );
        }

        $address = $customer->getAddressById((int)$addressId);

        if( ! $address->getId()) {
            return new MageQL_Core_Model_Customer_Result_Address_Update(
                MageQL_Core_Model_Customer_Result_Address_Update::ERROR_INVALID_ADDRESS_ID
            );
        }

        $modified = $form->setFields($address, $args["address"]);

        if( ! $modified) {
            return new MageQL_Core_Model_Customer_Result_Address_Update(
                MageQL_Core_Model_Customer_Result_Address_Update::NOT_MODIFIED,
                $customer,
                $address
            );
        }

        $result = $address->validate();

        if($result !== true) {
            return new MageQL_Core_Model_Customer_Result_Address_Update(
                MageQL_Core_Model_Customer_Result_Address_Update::ERROR_INVALID_ADDRESS,
                $customer,
                $address
            );
        }

        $address->save();

        Mage::dispatchEvent(self::EVENT_MUTATION_UPDATE_CUSTOMER_ADDRESS_SUCCESS, [
            "address" => $address,
            "context" => $ctx,
            "customer" => $customer,
        ]);

        return new MageQL_Core_Model_Customer_Result_Address_Update(
            MageQL_Core_Model_Customer_Result_Address_Update::SUCCESS,
            $customer,
            $address
        );
    }

    /**
     * @param mixed $unusedSrc
     */
    public static function mutateRemove(
        $unusedSrc,
        array $args
    ): MageQL_Core_Model_Customer_Result_Address_Remove {
        $session = Mage::getSingleton("customer/session");
        $addressId = $args["id"];
        $customer = $session->isLoggedIn() ? $session->getCustomer() : null;

        if( ! $customer) {
            return new MageQL_Core_Model_Customer_Result_Address_Remove(
                MageQL_Core_Model_Customer_Result_Address_Remove::ERROR_NOT_LOGGED_IN
            );
        }

        $address = $customer->getAddressById((int)$addressId);

        if( ! $address->getId()) {
            return new MageQL_Core_Model_Customer_Result_Address_Remove(
                MageQL_Core_Model_Customer_Result_Address_Remove::ERROR_INVALID_ADDRESS_ID
            );
        }

        $address->delete();

        return new MageQL_Core_Model_Customer_Result_Address_Remove(
            MageQL_Core_Model_Customer_Result_Address_Remove::SUCCESS
        );
    }
}
