<?php

declare(strict_types=1);

namespace MageQL\Schema;

use MageQL\Registry;

use MageQL\Type\AbstractBuilder;
use MageQL\Type\ArgumentBuilder;
use MageQL\Type\EnumBuilder;
use MageQL\Type\FieldBuilder;
use MageQL\Type\InputFieldBuilder;
use MageQL\Type\InputObjectBuilder;
use MageQL\Type\InterfaceBuilder;
use MageQL\Type\ObjectBuilder;

abstract class AbstractSchema implements SchemaInterface {
    /**
     * If it knows the supplied type a type builder for the supplied
     * type is returned, otherwise null.
     */
    abstract public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder;

    /**
     * Returns a map of the known fields for the supplied types, with
     * field-builders as values.
     *
     * @return Array<string, FieldBuilder|InputFieldBuilder>
     */
    abstract public function getTypeFields(string $typeName, Registry $registry): array;

    /**
     * Returns a list of type-names implementing interfaces which will not be
     * reachable when traversing the types normally.
     *
     * @return Array<string>
     */
    public function getUnreachableTypes(): array {
        return [];
    }

    /**
     * Creates a new object field argument description.
     */
    public function argument(string $type, string $description): ArgumentBuilder {
        return new ArgumentBuilder([
            "type" => $type,
            "description" => $description,
        ]);
    }

    /**
     * Creates a new object field description.
     */
    public function field(string $type, string $description): FieldBuilder {
        return new FieldBuilder([
            "type" => $type,
            "description" => $description,
        ]);
    }

    /**
     * Creates an enum type builder.
     */
    public function enum(string $description, array $values): EnumBuilder {
        return new EnumBuilder([
            "description" => $description,
            "values" => $values,
        ]);
    }

    /**
     * Creates a new type describing an object type.
     */
    public function inputObject(string $description): InputObjectBuilder {
        return new InputObjectBuilder([
            "description" => $description,
        ]);
    }

    /**
     * Creates a new object field description.
     */
    public function inputField(string $type, string $description): InputFieldBuilder {
        return new InputFieldBuilder([
            "type" => $type,
            "description" => $description,
        ]);
    }

    /**
     * Creates a new type describing an object type.
     */
    public function object(string $description): ObjectBuilder {
        return new ObjectBuilder([
            "description" => $description,
        ]);
    }

    /**
     * Creates a new type describing an interface type.
     */
    public function interface(string $description): InterfaceBuilder {
        return new InterfaceBuilder([
            "description" => $description,
        ]);
    }
}
