<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Schema\AbstractSchema;
use MageQL\Type\AbstractBuilder;
use MageQL\Registry;

class MageQL_Core_Model_Schema_Customer extends AbstractSchema {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "CustomerAddressInput":
            return $this->inputObject("Object containing a partially or fully populated address");

        case "CreateCustomerAddressResult":
            return $this->object("Type containing the result of createCustomerAddress")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "CreateCustomerAddressResultType":
            return $this->enum("Type of result from removeCustomerAddress mutation", [
                MageQL_Core_Model_Customer_Result_Address_Create::SUCCESS => [
                    "description" => "Address",
                ],
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_INVALID_ADDRESS => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "Customer":
            return $this->object("Type containing customer information")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "CustomerAddress":
            return $this->object("A customer address")
                ->setResolveField("MageQL\\defaultVarienObjectResolver")
                ->setInterfaces(["Address"]);

        case "UpdateCustomerInput":
            return $this->inputObject("New data for customer");

        case "LoginCustomerResult":
            return $this->object("Type containing the result of a login operation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "LoginCustomerResultType":
            return $this->enum("Type indicating the result of a login operation", [
                MageQL_Core_Model_Customer_Result_Login::SUCCESS => [
                    "description" => "A successful login",
                ],
                MageQL_Core_Model_Customer_Result_Login::ERROR_INVALID_LOGIN => [
                    "description" => "Invalid email or password.",
                ],
                MageQL_Core_Model_Customer_Result_Login::ERROR_NOT_CONFIRMED => [
                    "description" => "The account is not yet confirmed by email.",
                ],
                MageQL_Core_Model_Customer_Result_Login::ERROR_LOGIN_NOT_ENABLED => [
                    "description" => "Login mutation is not enabled",
                ],
            ]);

        case "RemoveCustomerAddressResult":
            return $this->object("Type containing the result of a customer address removal.")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "RemoveCustomerAddressResultType":
            return $this->enum("Type of result from removeCustomerAddress mutation", [
                MageQL_Core_Model_Customer_Result_Address_Remove::SUCCESS => [
                    "description" => "Address was successfully removed.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Remove::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Remove::ERROR_INVALID_ADDRESS_ID => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "SetCustomerDefaultAddressResult":
            return $this->object("Result of setCustomerDefaultShippingAddress or setCustomerDefaultBillingAddress.")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SetCustomerDefaultAddressResultType":
            return $this->enum("Type of result from updateCustomerEmail mutation", [
                MageQL_Core_Model_Customer_Result_Address_Default::SUCCESS => [
                    "description" => "Default address was successfully updated.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Default::NOT_MODIFIED => [
                    "description" => "Default address was not modified.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_INVALID_ADDRESS_ID => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "UpdateCustomerResult":
            return $this->object("Result of updateCustomer mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "UpdateCustomerResultType":
            return $this->enum("Type of result from updateCustomer mutation", [
                MageQL_Core_Model_Customer_Result_Update::SUCCESS => [
                    "description" => "Address was successfully updated",
                ],
                MageQL_Core_Model_Customer_Result_Update::NOT_MODIFIED => [
                    "description" => "Address was not modified",
                ],
                MageQL_Core_Model_Customer_Result_Update::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
            ]);

        case "UpdateCustomerAddressResult":
            return $this->object("Result of updateCustomerAddress mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "UpdateCustomerAddressResultType":
            return $this->enum("Type of result from updateCustomerAddress mutation", [
                MageQL_Core_Model_Customer_Result_Address_Update::SUCCESS => [
                    "description" => "Address was successfully updated",
                ],
                MageQL_Core_Model_Customer_Result_Address_Update::NOT_MODIFIED => [
                    "description" => "Address was not modified",
                ],
                MageQL_Core_Model_Customer_Result_Address_Update::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Update::ERROR_INVALID_ADDRESS_ID => [
                    "description" => "The supplied address id is invalid.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_INVALID_ADDRESS => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "UpdateCustomerEmailResult":
            return $this->object("Result of updateCustomerEmail mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "UpdateCustomerEmailResultType":
            return $this->enum("Type of result from updateCustomerEmail mutation", [
                MageQL_Core_Model_Customer_Result_UpdateEmail::SUCCESS => [
                    "description" => "Email was successfully updated.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::NOT_MODIFIED => [
                    "description" => "Email was not modified.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_INVALID_EMAIL_ADDRESS => [
                    "description" => "The supplied email address is invalid.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_EMAIL_EXISTS => [
                    "description" => "The supplied email is used by another customer",
                ],
            ]);

        default:
            return null;
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "CustomerAddressInput":
            return Mage::getSingleton("mageql/form_customer_address")->getInputFields();

        case "CreateCustomerAddressResult":
            return [
                "result" => $this->field("CreateCustomerAddressResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
                "address" => $this->field("CustomerAddress", "The set address customer address, if no error occurred or if address failed to validate"),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors if the address failed to validate"),
            ];

        case "Customer":
            $config = Mage::getSingleton("mageql/attributes_customer");
            $fields = $config->createFields($config->getAttributes());

            // TODO: Customer group
            return array_merge($fields, [
                "addresses" => $this->field("[CustomerAddress!]!", "List of addresses for the customer."),
                "email" => $this->field("String!", "Customer email"),
                "createdAt" => $this->field("String!", "Customer created date")
                    ->setResolver("MageQL_Core_Model_Customer::resolveCreatedAt"),
            ]);

        case "CustomerAddress":
            return array_merge($registry->getFieldBuilders("Address"), [
                "id" => $this->field("ID!", "Address id, used to update/delete"),
                "isDefaultBilling" => $this->field("Boolean!", "If the address is the default billing address")
                    ->setResolver("MageQL_Core_Model_Customer_Address::resolveIsDefaultBilling"),
                "isDefaultShipping" => $this->field("Boolean!", "If the address is the default shipping address")
                    ->setResolver("MageQL_Core_Model_Customer_Address::resolveIsDefaultShipping"),
            ]);

        case "LoginCustomerResult":
            return [
                "result" => $this->field("LoginCustomerResultType!", "The result type"),
                "customer" => $this->field("Customer", "Present if it is a success"),
            ];

        case "RemoveCustomerAddressResult":
            return [
                "result" => $this->field("RemoveCustomerAddressResultType!", "The type of result from removeCustomerAddress"),
            ];

        case "UpdateCustomerInput":
            return Mage::getSingleton("mageql/form_customer")->getInputFields();

        case "Mutation":
            return [
                // TODO: More customer mutations
                "loginCustomer" => $this->field("LoginCustomerResult!", "Attempts to log in a customer")
                    ->addArgument("email", $this->argument("String!", "The email of the customer account"))
                    ->addArgument("password", $this->argument("String!", "The password"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateLogin"),
                "logoutCustomer" => $this->field("Boolean!", "Attempts to log out a customer")
                    ->setResolver("MageQL_Core_Model_Customer::mutateLogout"),
                "updateCustomer" => $this->field("UpdateCustomerResult!", "Updates the customer information for the currently logged in customer")
                    ->addArgument("customer", $this->argument("UpdateCustomerInput!", "The new customer information"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateUpdate"),
                "updateCustomerEmail" => $this->field("UpdateCustomerEmailResult!", "Updates the email for the currently logged in customer, this will also propagate to any quote.")
                    ->addArgument("email", $this->argument("String!", "The new email for the account"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateUpdateEmail"),
                "setCustomerDefaultBillingAddress" => $this->field("SetCustomerDefaultAddressResult!", "Attempts to set the given address id as the default billing address for the currently logged in customer")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address to use as default billing address"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateSetDefaultBillingAddress"),
                "setCustomerDefaultShippingAddress" => $this->field("SetCustomerDefaultAddressResult!", "Attempts to set the given address id as the default shipping address for the currently logged in customer")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address to use as default shipping address"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateSetDefaultShippingAddress"),
                "createCustomerAddress" => $this->field("CreateCustomerAddressResult!", "Creates a customer address")
                    ->addArgument("address", $this->argument("CustomerAddressInput!", "The new address data"))
                    ->addArgument("setDefaultBilling", $this->argument("Boolean", "If the newly created address should be set as the default billing address")
                        ->setDefaultValue(false))
                    ->addArgument("setDefaultShipping", $this->argument("Boolean", "If the newly created address should be set as the default shipping address")
                        ->setDefaultValue(false))
                    ->setResolver("MageQL_Core_Model_Customer_Address::mutateCreate"),
                "updateCustomerAddress" => $this->field("UpdateCustomerAddressResult!", "Updates a customer address with new data")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address to update"))
                    ->addArgument("address", $this->argument("CustomerAddressInput!", "Address data to update"))
                    ->setResolver("MageQL_Core_Model_Customer_Address::mutateUpdate"),
                "removeCustomerAddress" => $this->field("RemoveCustomerAddressResult!", "Attempts to delete a customer address")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address"))
                    ->setResolver("MageQL_Core_Model_Customer_Address::mutateRemove"),
            ];

        case "Query":
            return [
                "customer" => $this->field("Customer", "Customer information if a customer is logged in")
                    ->setResolver("MageQL_Core_Model_Customer::resolveCustomer"),
            ];

        case "SetCustomerDefaultAddressResult":
            return [
                "result" => $this->field("SetCustomerDefaultAddressResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
                "address" => $this->field("Address", "The set address customer address, if no error occurred"),
            ];

        case "UpdateCustomerResult":
            return [
                "result" => $this->field("UpdateCustomerResultType!", "The result type"),
                "customer" => $this->field("Customer", "Present if it is a success"),
            ];

        case "UpdateCustomerAddressResult":
            return [
                "result" => $this->field("UpdateCustomerAddressResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
                "address" => $this->field("CustomerAddress", "The updated customer address, if no error occurred"),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors if the address failed to validate"),
            ];

        case "UpdateCustomerEmailResult":
            return [
                "result" => $this->field("UpdateCustomerEmailResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
            ];

        default:
            return [];
        }
    }
}
