<?php

declare(strict_types=1);

use GraphQL\Error\Warning;
use GraphQL\Type\Schema;

use MageQL\Registry;
use MageQL\Schema\DefaultSchema;
use MageQL\Schema\Federation;

class MageQL_Core_Helper_Data extends Mage_Core_Helper_Abstract {
    /**
     * Default schema name.
     */
    const SCHEMA_DEFAULT = "default";

    /**
     * Boolean configuration value to enable GraphQL Login.
     */
    const CONFIG_CUSTOMER_STARTUP_ENABLE_LOGIN = "customer/startup/graphql_enable_login";

    /**
     * Creates a dynamically expanding schema instance based on the context.
     *
     * Defaults:
     *
     *  * unreachable: false
     *  * federation: true
     *
     * @param array{ unreachable?: bool, federation?: bool } $include
     */
    public function loadSchema(
        MageQL_Core_Model_Context $context,
        array $include = []
    ): Schema {
        $schemaName = $context->getSchemaName();
        $config = $context->getConfig(sprintf("mageql/schema/%s", $schemaName));
        $factories = [];

        if( ! $config) {
            throw new MageQL_Core_Exception_SchemaNotFound($schemaName);
        }

        $factories[] = new DefaultSchema();

        // TODO: Do we reverse this so we can have priority?
        foreach($config as $k => $v) {
            if(array_key_exists("model", $v)) {
                /**
                 * @var false | MageQL_Core_Model_Schema_Abstract
                 */
                $model = Mage::getModel($v["model"], array_merge([
                    "name" => $k,
                    "context" => $context,
                ], $v));

                if( ! $model instanceof MageQL_Core_Model_Schema_Abstract) {
                    throw new Exception(sprintf(
                        "%s: Model instantiated from config '%s' must be an instance of %s",
                        __METHOD__,
                        sprintf("config/default/mageql/schema/%s/%s/model", $schemaName, $k),
                        MageQL_Core_Model_Schema_Abstract::class
                    ));
                }

                $factories[] = $model;

                continue;
            }

            throw new Exception(sprintf("%s: Missing model tag in 'mageql/schema/%s/%s'.", __METHOD__, $schemaName, $k));
        }

        if($include["federation"] ?? true) {
            $factories[] = new MageQL_Core_Model_Schema_Federation([
                "context" => $context,
            ]);
        }

        /**
         * In case we get warnings we need to keep track of them
         *
         * @var Array<array{id:string, msg:string}>
         */
        $warnings = [];

        if(Mage::getIsDeveloperMode()) {
            /**
             * Not sure why psalm fails to recognize that msg and id are used
             *
             * @psalm-suppress UnusedClosureParam
             */
            Warning::setWarningHandler(function(string $msg, int $id) use(&$warnings) {
                $warnings[] = [
                    "id"  => $id,
                    "msg" => $msg,
                ];
            });
        }

        $registry = new Registry($factories, [
            "includeUnreachable" => $include["unreachable"] ?? false,
        ]);

        $schema = $registry->getSchema();

        if( ! empty($warnings)) {
            // Assert the schema immediately, probably why we got a warning
            $schema->assertValid();

            throw new Exception(sprintf(
                "%s: Warnings: %s",
                __METHOD__,
                implode(", ", array_map(function(array $warning) {
                    return sprintf("%s: %s", $warning["id"], $warning["msg"]);
                }, $warnings))
            ));
        }

        if(Mage::getIsDeveloperMode()) {
            Warning::setWarningHandler(null);
        }

        return $schema;
    }
}
