<?php

declare(strict_types=1);

use MageQL\Type\InputFieldBuilder;

use function MageQL\snakeCaseToCamel;

abstract class MageQL_Core_Model_Form_Abstract extends Mage_Core_Model_Abstract {
    abstract public function getForm(): Mage_Eav_Model_Form;

    /**
     * @return Array<string, InputFieldBuilder>
     */
    public function getCustomFields(): array {
        return [];
    }

    public function includeAttribute(Mage_Eav_Model_Attribute $attribute): bool {
        return true;
    }

    public function createInputField(Mage_Eav_Model_Attribute $attribute): InputFieldBuilder  {
        return new InputFieldBuilder([
            "type" => $this->toGraphQLType($attribute),
            "description" => $attribute->getFrontendLabel(),
        ]);
    }

    public function toGraphQLType(Mage_Eav_Model_Attribute $attribute): string  {
        // FIXME: Make stuff required
        // $required = $attribute->getIsRequired() ? "!" : "";
        $required = "";

        switch($attribute->getFrontendInput()) {
        case "boolean":
            return "Boolean$required";
        case "multiselect":
            // Multiselect is an array of strings
            return "[String!]$required";
        case "text":
        case "select":
            return "String$required";
        default:
            switch($attribute->getBackendType()) {
            case "text":
            case "varchar":
            case "datetime":
                return "String$required";
            case "int":
                return "Int$required";
            case "decimal":
                return "Float$required";
            default:
                throw new Exception("Missing type-implementation for backend_type on '".json_encode($attribute)."'.");
            }
        }
    }

    protected function getAttributes(): array {
        $form = $this->getForm();

        return array_filter($form->getAttributes(), [$this, "includeAttribute"]);
    }

    /**
     * @return Array<InputFieldBuilder>
     */
    public function getInputFields(): array {
        $attributes = $this->getAttributes();

        return array_merge(array_combine(
            array_map("MageQL\snakeCaseToCamel", array_keys($attributes)),
            array_map([$this, "createInputField"], $attributes)
        ), $this->getCustomFields());
    }

    public function setFields(Varien_Object $obj, array $data): bool {
        $attributes = $this->getAttributes();
        $modified = false;

        foreach($attributes as $field => $attribute) {
            $name = ucfirst(snakeCaseToCamel($attribute->getAttributeCode()));
            $read = "get$name";
            $write = "set$name";

            if(array_key_exists($field, $data) && $data[$field] !== $obj->$read()) {
                $obj->$write($data[$field]);
                $modified = true;
            }
        }

        return $modified;
    }

    /**
     * Copies attribute $field from $from to $to, returning true if anything
     * was modified.
     */
    protected function copyDataField(
        Varien_Object $from,
        Varien_Object $to,
        string $field,
        bool $skipNull
    ): bool {
        $value = $from->getData($field);

        if(!$skipNull || $value !== null) {
            if($value !== $to->getData()) {
                $to->setData($field, $value);

                return true;
            }
        }

        return false;
    }

    /**
     * Copies data from one object to another, returning true if anything was
     * modified.
     */
    public function copyData(
        Varien_Object $from,
        Varien_Object $to,
        bool $skipNull = false
    ): bool {
        $modified = false;

        foreach(array_keys($this->getAttributes()) as $field) {
            if($this->copyDataField($from, $to, $field, $skipNull)) {
                $modified = true;
            }
        }

        return $modified;
    }
}
