<?php

declare(strict_types=1);

namespace MageQL\Core;

use Mage;
use Exception;
use Throwable;

use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

require_once __DIR__."/ExceptionTest/Schema.php";

class ExceptionTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/mageql/schema/default/TheTest");
    }

    public function onNotSuccessfulTest(Throwable $e): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/mageql/schema/default/TheTest");

        throw $e;
    }

    public function testMissingModel(): void {
        Config::setConfigPath("default/mageql/schema/default/TheTest/model", "the_missing/schema");

        MagentoManager::init();

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("MageQL_Core_Helper_Data::loadSchema: Model instantiated from config 'config/default/mageql/schema/default/TheTest/model' must be an instance of MageQL_Core_Model_Schema_Abstract");

        MagentoManager::runRequest(new Request("GET /graphql?query={somethingOdd}"));
    }

    public function testUndefinedMethod(): void {
        Config::setConfigPath("global/models/custom/class", "ExceptionTestModel");
        Config::setConfigPath("default/mageql/schema/default/TheTest/model", "custom/schema");

        MagentoManager::init();

        $res = MagentoManager::runRequest(new Request("GET /graphql?query={somethingOdd}"));
        $log = MagentoManager::getExceptionLog();

        $this->assertStringContainsString("Exception: We failed! in ", $log);
        $this->assertStringContainsString("Stack trace:", $log);

        $this->assertEquals(500, $res->getHttpResponseCode());
    }

    public function testErrorExtensions(): void {
        Mage::setIsDeveloperMode(false);
        Config::setConfigPath("global/models/custom/class", "ExceptionTestModel");
        Config::setConfigPath("default/mageql/schema/default/TheTest/model", "custom/schema");

        MagentoManager::init();

        $resp = MagentoManager::runRequest(new Request("GET /graphql?query={clientException}"));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(400, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
