<?php

declare(strict_types=1);

class MageQL_Core_Model_Address extends Mage_Core_Model_Abstract {
    const MISSING_CITY = "errorMissingCity";
    const MISSING_COUNTRY = "errorMissingCountry";
    const MISSING_FIRSTNAME = "errorMissingFirstname";
    const MISSING_LASTNAME = "errorMissingLastname";
    const MISSING_POSTCODE = "errorMissingPostcode";
    const MISSING_REGION = "errorMissingRegion";
    const MISSING_STREET = "errorMissingStreet";
    const MISSING_TELEPHONE = "errorMissingTelephone";

    /**
     * List of basic address fields and their method-variants.
     */
    // TODO: define the fields in XML like Magento does?
    const FIELDS = [
        "prefix" => "Prefix",
        "firstname" => "Firstname",
        "middlename" => "Middlename",
        "lastname" => "Lastname",
        "suffix" => "Suffix",
        "company" => "Company",
        "street" => "Street",
        "city" => "City",
        "postcode" => "Postcode",
        "telephone" => "Telephone",
    ];

    public static function resolveCountryCode(Mage_Directory_Model_Country $src): string {
        return $src->getCountryId();
    }

    public static function resolveCountry(
        Mage_Customer_Model_Address_Abstract $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx
    ): Mage_Directory_Model_Country {
        $model = $src->getCountryModel();

        if( ! $model->getCountryId()) {
            /**
             * This is a mandatory setting.
             *
             * @var string
             */
            $code = $ctx->getStore()->getConfig(Mage_Core_Helper_Data::XML_PATH_DEFAULT_COUNTRY);

            $model->loadByCode($code);
        }

        return $model;
    }

    public static function resolveRegion(Mage_Customer_Model_Address_Abstract $src): ?Mage_Directory_Model_Region {
        return $src->getRegionId() ? $src->getRegionModel() : null;
    }

    public static function resolveStreet(Mage_Customer_Model_Address_Abstract $src): array {
        $street = $src->getStreet();
        $parts = is_array($street) ? $street : explode("\n", $street);

        return array_values(array_filter(array_map("trim", $parts)));
    }

    public static function resolveValidationErrors(Mage_Customer_Model_Address_Abstract $address): array {
        $result = $address->validate();

        if($result === true) {
            return [];
        }

        $translation = self::getErrorTranslation();
        $errors = [];

        foreach($result as $err) {
            if(array_key_exists($err, $translation)) {
                $errors[] = $translation[$err];
            }
            else {
                throw new Exception(sprintf("%s: Unknown address validation error '%s'.", __METHOD__, $err));
            }
        }

        return $errors;
    }

    // TODO: Move to helper
    /**
     * Updates the fields common to both billing, shipping, and customer address.
     *
     * @param array{
     *   prefix?: string,
     *   firstname?: string,
     *   middlename?: string,
     *   lastname?: string,
     *   suffix?: string,
     *   company?: string,
     *   street?: Array<string>,
     *   city?: string,
     *   postcode?: string,
     *   regionCode?: string,
     *   countryCode?: string,
     *   telephone?: string,
     * } $data
     * @return bool If the address was modified
     */
    public static function setCommonFields(
        Mage_Customer_Model_Address_Abstract $address,
        array $data
    ): bool {
        $helper = Mage::helper("mageql/address");
        $modified = false;

        foreach(self::FIELDS as $field => $method) {
            $read = "get$method";
            $write = "set$method";

            switch($field) {
            case "street":
                if(array_key_exists("street", $data)) {
                    $newStreet = trim(implode("\n", $data[$field]));
                    $oldStreet = trim(implode("\n", (array)$address->getStreet()));

                    if($newStreet !== $oldStreet) {
                        // Since we have already trimmed it and imploded it we
                        // do no have to call implodeStreetAddress
                        $address->setStreet($newStreet);

                        $modified = true;
                    }
                }

                break;

            default:
                if(array_key_exists($field, $data) && $data[$field] !== $address->$read()) {
                    $address->$write($data[$field]);
                    $modified = true;
                }
            }
        }

        // Country before region, to be able to validate
        if(array_key_exists("countryCode", $data) && $data["countryCode"] !== $address->getCountryId()) {
            $approvedIds = $helper->getAllowedCountryCodes(Mage::app()->getStore());
            $countryId = strtoupper(trim($data["countryCode"]));

            if( ! in_array($countryId, $approvedIds)) {
                // We throw since the frontend should only allow the pre-approved countries
                throw new MageQL_Core_InvalidCountryCodeException($countryId);
            }

            $address->setCountryId($countryId);

            $modified = true;
        }

        if(array_key_exists("regionCode", $data) && $data["regionCode"] !== $address->getRegionCode()) {
            // TODO: Validate
            $address->setRegionCode($data["regionCode"]);
            // Reset
            $address->unsRegionId();
            $address->unsRegion();

            $modified = true;
        }

        return $modified;
    }

    /**
     * @return Array<string, string>
     */
    public static function getErrorTranslation(): array {
        return [
            Mage::helper("customer")->__("Please enter the city.") => self::MISSING_CITY,
            Mage::helper("customer")->__("Please enter the country.") => self::MISSING_COUNTRY,
            Mage::helper("customer")->__("Please enter the first name.") => self::MISSING_FIRSTNAME,
            Mage::helper("customer")->__("Please enter the last name.") => self::MISSING_LASTNAME,
            Mage::helper("customer")->__("Please enter the state/province.") => self::MISSING_REGION,
            Mage::helper("customer")->__("Please enter the street.") => self::MISSING_STREET,
            Mage::helper("customer")->__("Please enter the telephone number.") => self::MISSING_TELEPHONE,
            Mage::helper("customer")->__("Please enter the zip/postal code.") => self::MISSING_POSTCODE,
        ];
    }
}
