<?php

declare(strict_types=1);

use Fixtures\Customer;

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class CustomerTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/customer/startup/graphql_enable_login");

        $this->cleanup();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();
        Config::unsetConfigPath("default/customer/startup/graphql_enable_login");

        $this->cleanup();

        throw $e;
    }

    public function cleanup(): void {
        MagentoManager::init();

        $fixture = new Customer();

        $fixture->load();
    }

    protected function customerFixture() {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        return Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
    }

    public function testCustomerNotLoggedIn() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCustomerLoggedIn() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginSuccess() {
        $customer = $this->customerFixture();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-customer@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginFailure() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-fail@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginDisabled() {
        Config::setConfigPath("default/customer/startup/graphql_enable_login", "0");

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-fail@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLogout() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                logoutCustomer
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLogoutBatch() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            ["query" => 'mutation {
                logoutCustomer
            }'],
            ["query" => 'query {
                customer {
                    firstname
                    email
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmailNotLoggedIn() {
        $this->customerFixture();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailNotModified() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test-customer@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailInvalidEmail() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "testexample.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailExists() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test-duplicate@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmail() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test@example.com") {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateEmailQuote() {
        $customer = $this->customerFixture();

        $checkout = Mage::getSingleton("checkout/session");
        $quote = $checkout->getQuote();

        $quote->setCustomerEmail("another@example.com");
        $quote->save();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerEmail(email: "test@example.com") {
                        result
                    }
                }'));

            $this->assertEquals("test@example.com", $quote->getCustomerEmail());
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultShippingAddress() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $address = $customer->getDefaultBillingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($customer->getData("default_billing"), $customer->getData("default_shipping"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetDefaultShippingAddressNoop() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $address = $customer->getDefaultShippingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($address->getId(), $customer->getData("default_shipping"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultShippingAddressInvalid() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode("0").') {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultShippingAddressNotLoggedIn() {
        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultShippingAddress(id: '.json_encode("0").') {
                        result
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testSetDefaultBillingAddress() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $address = $customer->getDefaultShippingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultBillingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($customer->getData("default_billing"), $customer->getData("default_shipping"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testSetDefaultBillingAddressNoop() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $address = $customer->getDefaultBillingAddress();

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    setCustomerDefaultBillingAddress(id: '.json_encode($address->getId()).') {
                        result
                    }
                }'));

            $this->assertEquals($address->getId(), $customer->getData("default_billing"));
            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testRemoveCustomerAddress() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $address = $customer->getDefaultShippingAddress();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                removeCustomerAddress(id: '.json_encode($address->getId()).') {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveCustomerAddressInvalid() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                removeCustomerAddress(id: '.json_encode("0").') {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testRemoveCustomerAddressNotLoggedIn() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                removeCustomerAddress(id: '.json_encode("0").') {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateCustomerAddressNotLoggedIn() {
        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode("0").', address: {}) {
                        result
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerAddressNotFound() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode("0").', address: {}) {
                        result
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerAddressNotModified() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode($customer->getDefaultBillingAddress()->getId()).', address: {
                        firstname: "Test Firstname Billing",
                        lastname: "Test Lastname Billing",
                        city: "Test City Billing",
                        countryCode: "SE",
                        postcode: "54321",
                        street: "Test Street Billing",
                        telephone: "0987654321",
                    }) {
                        result
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerAddress() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    updateCustomerAddress(id: '.json_encode($customer->getDefaultBillingAddress()->getId()).', address: {
                        firstname: "Test Firstname Billing Modified",
                        lastname: "Test Lastname Billing",
                        city: "Test City Billing",
                        countryCode: "SE",
                        postcode: "12345",
                        street: "Test Street Billing",
                        telephone: "0987654321",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCreateCustomerAddress() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    createCustomerAddress(address: {
                        firstname: "Test Added",
                        lastname: "Test The Added ",
                        city: "Test City",
                        countryCode: "SE",
                        postcode: "11111",
                        street: "Test",
                        telephone: "0987654321",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCreateCustomerAddressNotLoggedIn() {
        try {
            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    createCustomerAddress(address: {
                        firstname: "Test Added",
                        lastname: "Test The Added ",
                        city: "Test City",
                        countryCode: "SE",
                        postcode: "11111",
                        street: "Test",
                        telephone: "0987654321",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testCreateCustomerAddressInvalid() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                    createCustomerAddress(address: {
                        firstname: "Test Added",
                        lastname: "Test The Added ",
                        city: "Test City",
                        street: "Test",
                        telephone: "0987654321",
                    }) {
                        result
                        address {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                        validationErrors
                    }
                }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerAddress_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_createCustomerAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }

        MagentoManager::reset();

        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateCustomer() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                updateCustomer(customer: { firstname: "UpdateFirstname", lastname: "UpdateLastname" }) {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        addresses {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertNotFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerNotModified() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        try {
            Varien_Profiler::enable();

            $resp = MagentoManager::runRequest(new Request("POST /graphql", [
                "Content-Type" => "application/graphql",
            ], 'mutation {
                updateCustomer(customer: { firstname: "Test", lastname: "Customer" }) {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                        addresses {
                            firstname
                            lastname
                            street
                            city
                            postcode
                            country {
                                code
                                name
                            }
                            telephone
                            isDefaultBilling
                            isDefaultShipping
                        }
                    }
                }
            }'));

            $this->assertMatchesJsonSnapshot($resp->getBody());
            $this->assertEquals(200, $resp->getHttpResponseCode());
            $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomer_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_updateCustomerEmail_success", null));
            $this->assertFalse(Varien_Profiler::fetch("DISPATCH EVENT:mageql_core_mutation_setCustomerDefaultAddress_success", null));
        }
        finally {
            Varien_Profiler::disable();
        }
    }

    public function testUpdateCustomerNotLoggedIn() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
            updateCustomer(customer: { firstname: "Test", lastname: "Customer" }) {
                result
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                    addresses {
                        firstname
                        lastname
                        street
                        city
                        postcode
                        country {
                            code
                            name
                        }
                        telephone
                        isDefaultBilling
                        isDefaultShipping
                    }
                }
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
