<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Type\AbstractBuilder;
use MageQL\Registry;

class MageQL_Core_Model_Schema_Default extends MageQL_Core_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "Address":
            return $this->interface("Base address properties");

        case "AddressInput":
            return $this->inputObject("Object containing a partially or fully populated address");

        case "AddressRegion":
            return $this->object("An address region")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "AddressValidationError":
            return $this->enum("Enum representing possible validation errors for an address", [
                MageQL_Core_Model_Address::MISSING_CITY => [
                    "description" => "City is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_COUNTRY => [
                    "description" => "Country is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_FIRSTNAME => [
                    "description" => "Firstname is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_LASTNAME => [
                    "description" => "Lastname is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_POSTCODE => [
                    "description" => "Postcode is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_REGION => [
                    "description" => "Region is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_STREET => [
                    "description" => "Street is missing a value",
                ],
                MageQL_Core_Model_Address::MISSING_TELEPHONE => [
                    "description" => "Telephone is missing a value",
                ],
            ]);

        case "Country":
            return $this->object("A country in an address")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "CreateCustomerAddressResult":
            return $this->object("Type containing the result of createCustomerAddress")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "CreateCustomerAddressResultType":
            return $this->enum("Type of result from removeCustomerAddress mutation", [
                MageQL_Core_Model_Customer_Result_Address_Create::SUCCESS => [
                    "description" => "Address",
                ],
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_INVALID_ADDRESS => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "Customer":
            return $this->object("Type containing customer information")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "CustomerAddress":
            return $this->object("A customer address")
                ->setResolveField("MageQL\\defaultVarienObjectResolver")
                ->setInterfaces(["Address"]);

        case "ImageData":
            return $this->object("Image data");

        case "LoginCustomerResult":
            return $this->object("Type containing the result of a login operation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "LoginCustomerResultType":
            return $this->enum("Type indicating the result of a login operation", [
                MageQL_Core_Model_Customer_Result_Login::SUCCESS => [
                    "description" => "A successful login",
                ],
                MageQL_Core_Model_Customer_Result_Login::ERROR_INVALID_LOGIN => [
                    "description" => "Invalid email or password.",
                ],
                MageQL_Core_Model_Customer_Result_Login::ERROR_NOT_CONFIRMED => [
                    "description" => "The account is not yet confirmed by email.",
                ],
            ]);

        case "Price":
            return $this->interface("Price information which carries tax information");

        case "RemoveCustomerAddressResult":
            return $this->object("Type containing the result of a customer address removal.")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "RemoveCustomerAddressResultType":
            return $this->enum("Type of result from removeCustomerAddress mutation", [
                MageQL_Core_Model_Customer_Result_Address_Remove::SUCCESS => [
                    "description" => "Address was successfully removed.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Remove::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Remove::ERROR_INVALID_ADDRESS_ID => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "RouteType":
            return $this->enum("Type indicating variant of route resource", [
                "category" => [
                    "description" => "A category",
                ],
                "cms_page" => [
                    "description" => "A CMS page",
                ],
                "product" => [
                    "description" => "A product",
                ],
                "redirect" => [
                    "description" => "A redirect to another path",
                ],
            ]);

        case "RouteRedirect":
            return $this->object("A response containing a category")
                ->setInterfaces(["Route"]);

        case "Route":
            return $this->interface("Response from a route")
                ->setTypeResolver("MageQL_Core_Model_Route::typeResolver");

        case "SetCustomerDefaultAddressResult":
            return $this->object("Result of setCustomerDefaultShippingAddress or setCustomerDefaultBillingAddress.")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "SetCustomerDefaultAddressResultType":
            return $this->enum("Type of result from updateCustomerEmail mutation", [
                MageQL_Core_Model_Customer_Result_Address_Default::SUCCESS => [
                    "description" => "Default address was successfully updated.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Default::NOT_MODIFIED => [
                    "description" => "Default address was not modified.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Default::ERROR_INVALID_ADDRESS_ID => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "StoreInfo":
            return $this->object("Basic store information")
                ->setResolveField("MageQL\\defaultVarienObjectResolver");

        case "UpdateCustomerAddressResult":
            return $this->object("Result of updateCustomerAddress mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "UpdateCustomerAddressResultType":
            return $this->enum("Type of result from updateCustomerAddress mutation", [
                MageQL_Core_Model_Customer_Result_Address_Update::SUCCESS => [
                    "description" => "Address was successfully updated",
                ],
                MageQL_Core_Model_Customer_Result_Address_Update::NOT_MODIFIED => [
                    "description" => "Address was not modified",
                ],
                MageQL_Core_Model_Customer_Result_Address_Update::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Update::ERROR_INVALID_ADDRESS_ID => [
                    "description" => "The supplied address id is invalid.",
                ],
                MageQL_Core_Model_Customer_Result_Address_Create::ERROR_INVALID_ADDRESS => [
                    "description" => "The supplied address id is invalid.",
                ],
            ]);

        case "UpdateCustomerEmailResult":
            return $this->object("Result of updateCustomerEmail mutation")
                ->setResolveField("MageQL\\defaultMethodResolver");

        case "UpdateCustomerEmailResultType":
            return $this->enum("Type of result from updateCustomerEmail mutation", [
                MageQL_Core_Model_Customer_Result_UpdateEmail::SUCCESS => [
                    "description" => "Email was successfully updated.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::NOT_MODIFIED => [
                    "description" => "Email was not modified.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_NOT_LOGGED_IN => [
                    "description" => "Customer is not logged in.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_INVALID_EMAIL_ADDRESS => [
                    "description" => "The supplied email address is invalid.",
                ],
                MageQL_Core_Model_Customer_Result_UpdateEmail::ERROR_EMAIL_EXISTS => [
                    "description" => "The supplied email is used by another customer",
                ],
            ]);
        }

        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "Address":
            return [
                "prefix" => $this->field("String", "Name prefix"),
                "firstname" => $this->field("String", "Firstname"),
                "middlename" => $this->field("String", "Name prefix"),
                "lastname" => $this->field("String", "Lastname"),
                "suffix" => $this->field("String", "Name suffix"),
                "company" => $this->field("String", "Company"),
                "street" => $this->field("[String!]!", "Street")
                    ->setResolver("MageQL_Core_Model_Address::resolveStreet"),
                "city" => $this->field("String", "City"),
                "postcode" => $this->field("String", "Postcode"),
                "region" => $this->field("AddressRegion", "Region")
                    ->setResolver("MageQL_Core_Model_Address::resolveRegion"),
                "country" => $this->field("Country!", "Country")
                    ->setResolver("MageQL_Core_Model_Address::resolveCountry"),
                "telephone" => $this->field("String", "Telephone"),
            ];

        case "AddressInput":
            return [
                "prefix" => $this->inputField("String", "Name prefix"),
                "firstname" => $this->inputField("String", "Firstname"),
                "middlename" => $this->inputField("String", "Name prefix"),
                "lastname" => $this->inputField("String", "Lastname"),
                "suffix" => $this->inputField("String", "Name suffix"),
                "company" => $this->inputField("String", "Company"),
                "street" => $this->inputField("[String!]", "Street, one line per array item"),
                "city" => $this->inputField("String", "City"),
                "postcode" => $this->inputField("String", "Postcode"),
                "regionCode" => $this->inputField("ID", "Region code"),
                "countryCode" => $this->inputField("ID", "Country code"),
                "telephone" => $this->inputField("String", "Telephone"),
            ];

        case "AddressRegion":
            return [
                "code" => $this->field("String!", "Region code, ISO 3166-2"),
                "name" => $this->field("String!", "Region name"),
            ];

        case "Country":
            return [
                "code" => $this->field("ID!", "Country code, ISO 3166-2")
                   ->setResolver("MageQL_Core_Model_Address::resolveCountryCode"),
               "name" => $this->field("String!", "Country name for the current locale"),
            ];

        case "CreateCustomerAddressResult":
            return [
                "result" => $this->field("CreateCustomerAddressResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
                "address" => $this->field("CustomerAddress", "The set address customer address, if no error occurred or if address failed to validate"),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors if the address failed to validate"),
            ];

        case "Customer":
            // Name, description and everything:
            $config = Mage::getSingleton("mageql/attributes_customer");
            // Category index everything, including all attribute sets
            $attrs = $config->getAttributes();
            $fields = $config->createFields($attrs);

            // TODO: Customer group
            // TODO: Orders for order history
            return array_merge($fields, [
                "addresses" => $this->field("[CustomerAddress!]", "List of addresses for the customer.")
                    ->setResolver("MageQL_Core_Model_Customer::resolveAddresses"),
                "email" => $this->field("String!", "Customer email"),
                "createdAt" => $this->field("String!", "Customer created date")
                    ->setResolver("MageQL_Core_Model_Customer::resolveCreatedAt"),
            ]);

        case "CustomerAddress":
            return array_merge($registry->getFieldBuilders("Address"), [
                "id" => $this->field("ID!", "Address id, used to update/delete"),
                "isDefaultBilling" => $this->field("Boolean!", "If the address is the default billing address")
                    ->setResolver("MageQL_Core_Model_Customer_Address::resolveIsDefaultBilling"),
                "isDefaultShipping" => $this->field("Boolean!", "If the address is the default shipping address")
                    ->setResolver("MageQL_Core_Model_Customer_Address::resolveIsDefaultShipping"),
            ]);

        case "RemoveCustomerAddressResult":
            return [
                "result" => $this->field("RemoveCustomerAddressResultType!", "The type of result from removeCustomerAddress"),
            ];

        case "ImageData":
            return [
                "src" => $this->field("String!", "The image URL")
                    ->addArgument("width", $this->argument("Int", "Maximum width of the image"))
                    ->addArgument("height", $this->argument("Int", "Minimum width of the image"))
                    ->addArgument(
                        "fill",
                        $this->argument("Boolean", "If to fill the image to the given size")
                             ->setDefaultValue(false)
                    )->setResolver("MageQL_Core_Model_Attributes_Image_Abstract::resolveSrc"),
            ];

        case "LoginCustomerResult":
            return [
                "result" => $this->field("LoginCustomerResultType!", "The result type"),
                "customer" => $this->field("Customer", "Present if it is a success"),
            ];

        case "Mutation":
            return [
                // TODO: More customer mutations
                "loginCustomer" => $this->field("LoginCustomerResult!", "Attempts to log in a customer")
                    ->addArgument("email", $this->argument("String!", "The email of the customer account"))
                    ->addArgument("password", $this->argument("String!", "The password"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateLogin"),
                "logoutCustomer" => $this->field("Boolean!", "Attempts to log out a customer")
                    ->setResolver("MageQL_Core_Model_Customer::mutateLogout"),
                "updateCustomerEmail" => $this->field("UpdateCustomerEmailResult!", "Updates the email for the currently logged in customer, this will also propagate to any quote.")
                    ->addArgument("email", $this->argument("String!", "The new email for the account"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateUpdateEmail"),
                "setCustomerDefaultBillingAddress" => $this->field("SetCustomerDefaultAddressResult!", "Attempts to set the given address id as the default billing address for the currently logged in customer")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address to use as default billing address"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateSetDefaultBillingAddress"),
                "setCustomerDefaultShippingAddress" => $this->field("SetCustomerDefaultAddressResult!", "Attempts to set the given address id as the default shipping address for the currently logged in customer")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address to use as default shipping address"))
                    ->setResolver("MageQL_Core_Model_Customer::mutateSetDefaultShippingAddress"),
                "createCustomerAddress" => $this->field("CreateCustomerAddressResult!", "Creates a customer address")
                    ->addArgument("address", $this->argument("AddressInput!", "The new address data"))
                    ->addArgument("setDefaultBilling", $this->argument("Boolean", "If the newly created address should be set as the default billing address")
                        ->setDefaultValue(false))
                    ->addArgument("setDefaultShipping", $this->argument("Boolean", "If the newly created address should be set as the default shipping address")
                        ->setDefaultValue(false))
                    ->setResolver("MageQL_Core_Model_Customer_Address::mutateCreate"),
                "updateCustomerAddress" => $this->field("UpdateCustomerAddressResult!", "Updates a customer address with new data")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address to update"))
                    ->addArgument("address", $this->argument("AddressInput!", "Address data to update"))
                    ->setResolver("MageQL_Core_Model_Customer_Address::mutateUpdate"),
                "removeCustomerAddress" => $this->field("RemoveCustomerAddressResult!", "Attempts to delete a customer address")
                    ->addArgument("id", $this->argument("ID!", "The id of the customer address"))
                    ->setResolver("MageQL_Core_Model_Customer_Address::mutateRemove"),
            ];

        case "Price":
            return [
                "exVat" => $this->field("Float!", "Price excluding VAT")
                    ->setResolver("MageQL_Core_Model_Price_Abstract::resolveExVat"),
                "incVat" => $this->field("Float!", "Price including VAT")
                    ->setResolver("MageQL_Core_Model_Price_Abstract::resolveIncVat"),
                "vat" => $this->field("Float!", "VAT amount")
                    ->setResolver("MageQL_Core_Model_Price_Abstract::resolveVat"),
            ];

        case "Query":
            return [
                "customer" => $this->field("Customer", "Customer information if a customer is logged in")
                    ->setResolver("MageQL_Core_Model_Customer::resolveCustomer"),
                "info" => $this->field("StoreInfo!", "Information about the current store")
                    ->setResolver("MageQL\\forwardResolver"),
                "route" => $this->field("Route", "Attempt to fetch a resource by its route")
                    ->addArgument("path", $this->argument("String!", "Path to resource"))
                    ->setResolver("MageQL_Core_Model_Route::resolve"),
            ];

        case "RouteRedirect":
            return [
                "isPermanent" => $this->field("Boolean!", "If the redirect is a permanent redirect")
                    ->setResolver("MageQL_Core_Model_Route_Redirect::resolveIsPermanent"),
                "type" => $this->field("RouteType!", "Type of route")
                    ->setResolver("MageQL_Core_Model_Route_Redirect::resolveType"),
                "url" => $this->field("String!", "Redirect to")
                    ->setResolver("MageQL_Core_Model_Route_Redirect::resolveUrl"),
            ];

        case "Route":
            return [
                "type" => $this->field("RouteType!", "Type of route resource"),
            ];

        case "SetCustomerDefaultAddressResult":
            return [
                "result" => $this->field("SetCustomerDefaultAddressResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
                "address" => $this->field("Address", "The set address customer address, if no error occurred"),
            ];

        case "StoreInfo":
            return [
                "baseCurrencyCode" => $this->field("String!", "Base currency code"),
                "baseUrl" => $this->field("String!", "Base URL"),
                "currencyCodes" => $this->field("[String!]!", "Currency codes")
                    ->setResolver("MageQL_Core_Model_Store::resolveCurrencyCodes"),
                "countries" => $this->field("[Country!]!", "List of available countries for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveCountries"),
                "defaultCountry" => $this->field("Country!", "Default country for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultCountry"),
                "defaultDescription" => $this->field("String", "Default page description")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultDescription"),
                "defaultTitle" => $this->field("String", "Default page title")
                    ->setResolver("MageQL_Core_Model_Store::resolveDefaultTitle"),
                "locale" => $this->field("String!", "Locale code for the store")
                    ->setResolver("MageQL_Core_Model_Store::resolveLocale"),
                "name" => $this->field("String!", "Name"),
                "titlePrefix" => $this->field("String", "Page title prefix")
                    ->setResolver("MageQL_Core_Model_Store::resolveTitlePrefix"),
                "titleSuffix" => $this->field("String", "Page title suffix")
                    ->setResolver("MageQL_Core_Model_Store::resolveTitleSuffix"),
            ];

        case "UpdateCustomerAddressResult":
            return [
                "result" => $this->field("UpdateCustomerAddressResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
                "address" => $this->field("CustomerAddress", "The updated customer address, if no error occurred"),
                "validationErrors" => $this->field("[AddressValidationError!]!", "List of validation errors if the address failed to validate"),
            ];

        case "UpdateCustomerEmailResult":
            return [
                "result" => $this->field("UpdateCustomerEmailResultType!", "The type of result"),
                "customer" => $this->field("Customer", "The customer result, if no error occurred"),
            ];
        }

        return [];
    }

    public function getUnreachableTypes(): array {
        return [
            "Price",
            "RouteRedirect",
        ];
    }
}
