<?php

declare(strict_types=1);

use Fixtures\Customer;

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;

class CustomerTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        $this->cleanup();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        $this->cleanup();

        throw $e;
    }

    public function cleanup(): void {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        // Reset the existing customer email
        $customer = Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test@example.com");

        if($customer->getId()) {
            $customer
                ->setEmail("test-customer@example.com")
                ->save();
        }
    }

    protected function customerFixture() {
        MagentoManager::init();

        $store = Mage::app()->getStore(MagentoManager::TESTING_STORE);

        return Mage::getModel("customer/customer")
            ->setWebsiteId($store->getWebsiteId())
            ->loadByEmail("test-customer@example.com");
    }

    public function testCustomerNotLoggedIn() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                email
                prefix
                firstname
                middlename
                lastname
                suffix
            }
        }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testCustomerLoggedIn() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'query {
                customer {
                    email
                    prefix
                    firstname
                    middlename
                    lastname
                    suffix
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginSuccess() {
        $customer = $this->customerFixture();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-customer@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLoginFailure() {
        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                loginCustomer(email: "test-fail@example.com", password: "test-customer") {
                    result
                    customer {
                        email
                        prefix
                        firstname
                        middlename
                        lastname
                        suffix
                    }
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLogout() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                logoutCustomer
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testLogoutBatch() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/json",
        ], json_encode([
            ["query" => 'mutation {
                logoutCustomer
            }'],
            ["query" => 'query {
                customer {
                    firstname
                    email
                }
            }'],
        ])));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmailNotLoggedIn() {
        $this->customerFixture();

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                updateCustomerEmail(email: "test@example.com") {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmailNotModified() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                updateCustomerEmail(email: "test-customer@example.com") {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmailInvalidEmail() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                updateCustomerEmail(email: "testexample.com") {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmailExists() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                updateCustomerEmail(email: "test-duplicate@example.com") {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmail() {
        $customer = $this->customerFixture();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                updateCustomerEmail(email: "test@example.com") {
                    result
                }
            }'));

        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }

    public function testUpdateEmailQuote() {
        $customer = $this->customerFixture();

        $checkout = Mage::getSingleton("checkout/session");
        $quote = $checkout->getQuote();

        $quote->setCustomerEmail("another@example.com");
        $quote->save();

        $session = Mage::getSingleton("customer/session");
        $session->login("test-customer@example.com", "test-customer");
        $session->setCustomerAsLoggedIn($customer);
        $customer->setSession($session->getSessionId());

        $resp = MagentoManager::runRequest(new Request("POST /graphql", [
            "Content-Type" => "application/graphql",
        ], 'mutation {
                updateCustomerEmail(email: "test@example.com") {
                    result
                }
            }'));

        $this->assertEquals("test@example.com", $quote->getCustomerEmail());
        $this->assertMatchesJsonSnapshot($resp->getBody());
        $this->assertEquals(200, $resp->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $resp->getHeader("Content-Type"));
    }
}
