<?php

declare(strict_types=1);

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

use Spatie\Snapshots\MatchesSnapshots;

use PHPUnit\Framework\TestCase;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use Crossroads\Magento\Test\Integration\Config;

class ExceptionTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest($e): void {
        MagentoManager::logQueries();

        throw $e;
    }

    public function testMissingModel() {
        Config::addConfig([
            [
                "name" => "default",
                "children" => [
                    [
                        "name" => "mageql",
                        "children" => [
                            [
                                "name" => "schema",
                                "children" => [
                                    [
                                        "name" => "default",
                                        "children" => [
                                            [
                                                "name" => "TheTest",
                                                "children" => [
                                                    [
                                                        "name" => "model",
                                                        "value" => "the_missing/schema",
                                                    ],
                                                ],
                                            ],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
        ]);

        MagentoManager::init();

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("MageQL_Core_Router_GraphQL::loadSchema: Model instantiated from config 'config/default/mageql/schema/default/TheTest/model' must be an instance of MageQL_Core_Model_Schema_Abstract");

        MagentoManager::runRequest(new Request("GET /graphql?query={somethingOdd}"));
    }

    public function testUndefinedMethod() {
        Config::addConfig([
            [
                "name" => "global",
                "children" => [
                    [
                        "name" => "models",
                        "children" => [
                            [
                                "name" => "custom",
                                "children" => [
                                    [
                                        "name" => "class",
                                        "value" => "ExceptionTestModel",
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
            [
                "name" => "default",
                "children" => [
                    [
                        "name" => "mageql",
                        "children" => [
                            [
                                "name" => "schema",
                                "children" => [
                                    [
                                        "name" => "default",
                                        "children" => [
                                            [
                                                "name" => "TheTest",
                                                "children" => [
                                                    [
                                                        "name" => "model",
                                                        "value" => "custom/schema",
                                                    ],
                                                ],
                                            ],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
        ]);

        MagentoManager::init();

        $res = MagentoManager::runRequest(new Request("GET /graphql?query={somethingOdd}"));
        $log = MagentoManager::getExceptionLog();

        $this->assertStringContainsString("Exception: We failed! in ", $log);
        $this->assertStringContainsString("Stack trace:", $log);

        $this->assertEquals(500, $res->getHttpResponseCode());
    }
}

class ExceptionTestModel_Schema extends MageQL_Core_Model_Schema_Abstract {
    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        return null;
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "Query":
            return [
                "somethingOdd" => $this->field("String!", "This crashes when resolved")
                    ->setResolver(function() {
                        throw new Exception("We failed!");
                    }),
            ];
        default:
            return [];
        }
    }
}
