<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Type\FieldBuilder;
use MageQL\Type\ArgumentBuilder;

use function MageQL\defaultVarienObjectResolver;
use function MageQL\snakeCaseToCamel;
use function MageQL\spacedToCamel;
use function MageQL\camelToSnakeCase;

/**
 * @psalm-type AttrSet array{
 *   id:int,
 *   name:string,
 *   type:string
 * }
 * @template Item of Varien_Object
 * @template AttrData of array{
 *   code:string,
 *   backend_type:string,
 *   input:string,
 *   label:string,
 *   required:bool,
 *   not_system:bool,
 *   attribute_set:Array<string>,
 *   attribute_set_type:Array<string>
 * }
 */
abstract class MageQL_Core_Model_Attributes_Abstract extends Mage_Core_Model_Abstract {
    /**
     * The unknown set, returned when a set cannot be found for an attribute.
     */
    const UNKNOWN_SET = [
        "id" => -1,
        "name" => "Unknown",
        "type" => "Unknown",
    ];

    /**
     * Map of attribute field-name => attribute data.
     *
     * Data:
     *
     *  * code: string, attribute code
     *  * backend_type: string
     *  * input: string
     *  * label: string
     *  * required: bool, if it must exist
     *  * not_system: bool, if it is not included in every attribute-set (aka system-attribute)
     *  * attribute_set: Array of string, list of all attribute sets the attribute is present in
     *  * attribute_set_type: Array of string, list of all attribute sets in camel case
     *
     * @var ?Array<AttrData>
     */
    protected $attributeData = null;

    /**
     * Map of attribute-set id to attribute set metadata.
     *
     * Data:
     *
     *  * id: int
     *  * name: string
     *  * type: string, camelized version of name
     *
     * @var ?Array<AttrSet>
     */
    protected $attributeSets = null;

    /**
     * Returns the entity type to be used for the attributes.
     */
    abstract protected function getEntityType(): string;

    /**
     * Returns a list of base attributes to always include when determining
     * used attributes.
     */
    abstract protected function getBaseAttributes(): array;

    /**
     * Returns a map of field name => list of attribute codes for fields which
     * do not have a corresponding attribute or do not  match their specific
     * attribute codes or require additional attributes to be able to resolve.
     */
    abstract protected function getFieldAttributeMap(): array;

    /**
     * Resizes a given image, returning the URL to it.
     *
     * @param Item $src
     * @param array{width?:int, height?:int, fill?:bool} $args
     */
    abstract public function resizeImage($src, string $attrCode, string $image, array $args): string;

    /**
     * Loads the name nad id of all attribute sets for the current entity type.
     *
     * @return Array<array{id:string, name:string}>
     */
    protected function loadAttributeSets(): array {
        $db = Mage::getSingleton("core/resource")->getConnection("core_read");

        return $db->query(
"SELECT
    s.attribute_set_id id,
    s.attribute_set_name name
FROM eav_attribute_set s
JOIN eav_entity_type t ON t.entity_type_id = s.entity_type_id
WHERE t.entity_type_code = ?", [$this->getEntityType()])->fetchAll();
    }

    /**
     * @return Array<AttrSet>
     */
    public function getSetData(): array {
        if($this->attributeSets === null) {
            $this->attributeSets = [];

            foreach($this->loadAttributeSets() as $set) {
                $this->attributeSets[(int)$set["id"]] = [
                    "id" => (int)$set["id"],
                    "name" => $set["name"],
                    "type" => spacedToCamel($set["name"]),
                ];
            }
        }

        return $this->attributeSets;
    }

    /**
     * Returns a list of type names for all attribute sets.
     *
     * @return Array<string>
     */
    public function getSetTypes(): array {
        $sets = [];

        foreach($this->getSetData() as $set) {
            $sets[] = $set["type"];
        }

        return $sets;
    }

    /**
     * Returns the attribute set with the given id.
     *
     * Returns a map with:
     *
     *  * name: string, "Unknown"  if not found
     *  * id: int, -1 if not found
     *
     * @return AttrSet
     */
    public function getSetById(int $attributeSetId): array {
        $data = $this->getSetData();

        return $data[$attributeSetId] ?? self::UNKNOWN_SET;
    }

    /**
     * Returns the attribute set name given an attribute set type.
     *
     * Returns a map with:
     *
     *  * name: string, "Unknown"  if not found
     *  * type: string, "Unknown"  if not found
     *  * id: int, -1 if not found
     *
     * @return AttrSet
     */
    public function getSetByType(string $attributeSetType): array {
        foreach($this->getSetData() as $set) {
            if($set["type"] === $attributeSetType) {
                return $set;
            }
        }

        return self::UNKNOWN_SET;
    }

    /**
     * Loads attribute data.
     *
     * Base fields:
     *
     *  * code: string
     *  * backend_type: string
     *  * input: string
     *  * label: string
     *  * required: bool
     *  * not_system: bool
     *  * attribute_set: comma-separated string
     */
    protected function loadAttributes(): array {
        $db = Mage::getSingleton("core/resource")->getConnection("core_read");

        return $db->query(
"SELECT a.attribute_code code,
    a.backend_type,
    a.frontend_input input,
    a.frontend_label label,
    a.is_required required,
    a.is_user_defined not_system,
    COALESCE(GROUP_CONCAT(DISTINCT s.attribute_set_name SEPARATOR ','), '') attribute_set
FROM eav_attribute a
JOIN eav_entity_type t ON t.entity_type_id = a.entity_type_id
JOIN eav_entity_attribute e ON e.attribute_id = a.attribute_id
JOIN eav_attribute_set s ON s.attribute_set_id = e.attribute_set_id
WHERE t.entity_type_code = ?
  AND a.backend_type <> 'static'
GROUP BY a.attribute_id
ORDER BY a.attribute_code ASC", [$this->getEntityType()])->fetchAll();
    }

    /**
     * Modifies the attribute data to their correct datatypes.
     *
     * @return AttrData
     */
    protected function filterAttributeData(array $a): array {
        $a["required"] = (bool)$a["required"];
        $a["not_system"] = $a["not_system"] ? true : false;
        $a["attribute_set"] = array_map("trim", explode(",", $a["attribute_set"]));
        $a["attribute_set_type"] = array_map("MageQL\\spacedToCamel", $a["attribute_set"]);

        return $a;
    }

    /**
     * Returns a map of field-name -> attribute data.
     *
     * @return Array<string, AttrData>
     */
    public function getAttributes(): array {
        if($this->attributeData === null) {
            $this->attributeData = [];

            foreach($this->loadAttributes() as $a) {
                $field = snakeCaseToCamel($a["code"]);

                $this->attributeData[$field] = $this->filterAttributeData($a);
            }
        }

        return $this->attributeData;
    }

    /**
     * @param Array<string, array{code:string}> $attributes Map of key => ["code" => attribute_code]
     * @param Array<string, mixed> $requestedFields Map of key => annything of fields to load
     *
     * @return Array<string>
     */
    public function getUsedAttributes(array $attributes, array $requestedFields): array {
        $toLoad = $this->getBaseAttributes();
        $nonAttrs = array_intersect_key($this->getFieldAttributeMap(), $requestedFields);
        $loadable = array_intersect_key($attributes, $requestedFields);

        foreach(array_keys($requestedFields) as $f) {
            if(array_key_exists($f, $nonAttrs)) {
                $toLoad = array_merge($toLoad, $nonAttrs[$f]);
            }

            if(array_key_exists($f, $loadable)) {
                $toLoad[] = $loadable[$f]["code"];
            }
        }

        return array_unique($toLoad);
    }

    /**
     * Builds a map of field-names -> GraphQL-types of the given map of
     * field-name -> attribute data.
     *
     * @param Array<string, AttrData> $attributes
     * @return Array<FieldBuilder>
     */
    public function createFields(array $attributes): array {
        return array_map([$this, "createFieldBuilder"], $attributes);
    }

    /**
     * Creates a field builder for the given field $a.
     *
     * @param AttrData $a
     */
    public function createFieldBuilder(array $a): FieldBuilder {
        $builder = new FieldBuilder([
            "type" => $this->toGraphQLType($a),
            "description" => $a["label"],
            "resolver" => $this->getFieldResolver($a),
        ]);

        switch($a["input"]) {
        case "image":
        case "media_image":
            $builder->addArgument("width", new ArgumentBuilder([
                "type" => "Int",
                "description" => "Maximum width of the image",
            ]));
            $builder->addArgument("height", new ArgumentBuilder([
                "type" => "Int",
                "description" => "Minimum width of the image",
            ]));
            $builder->addArgument("fill", new ArgumentBuilder([
                "type" => "Boolean",
                "description" => "If to fill the image to the given size",
                "defaultValue" => false,
            ]));
        }

        return $builder;
    }

    /**
     * Returns the GrqphQL type for the supplied attribute.
     *
     * @param AttrData $attr
     */
    public function toGraphQLType(array $attr): string {
        $required = $attr["required"] ? "!" : "";

        switch($attr["input"]) {
        case "boolean":
            return "Boolean$required";
        case "image":
        case "media_image":
            return "String$required";
        case "multiselect":
            // Multiselect is an array of strings
            return "[String!]$required";
        case "select":
            // TODO: How to handle boolean?
            return "String$required";
        // case "datetime":
        // case "date":
        // case "multiline":
        // case "price":
        // case "select":
        // case "text":
        // case "textarea":
        // case "weight":
        default:
            switch($attr["backend_type"]) {
            case "text":
            case "varchar":
            case "datetime":
                return "String$required";
            case "int":
                return "Int$required";
            case "decimal":
                return "Float$required";
            default:
                throw new Exception("Missing type-implementation for backend_type on '".json_encode($attr)."'.");
            }
        }
    }

    /**
     * Returns a callable field resolver for the given attribute, null if to
     * use default (eg. string).
     *
     * @param AttrData $a
     */
    public function getFieldResolver(array $a): ?callable {
        switch($a["input"]) {
        case "image":
        case "media_image":
            return [$this, "resolveImageAttribute"];
        case "date":
        case "datetime":
            return "MageQL\dateAttributeResolver";
        case "select":
            return "MageQL\selectAttributeResolver";
        case "multiselect":
            return "MageQL\multiselectAttributeResolver";
        default:
            if($a["backend_type"] === "datetime") {
                return "MageQL\dateAttributeResolver";
            }

            return null;
        }
    }

    /**
     * Resolver for image-attributes.
     *
     * @param Item $src
     * @param array{width?:int, height?:int, fill?:bool} $args
     * @return ?string
     */
    public function resolveImageAttribute(
        $src,
        array $args,
        MageQL_Core_Model_Context $context,
        ResolveInfo $info
    ): ?string {
        $value = defaultVarienObjectResolver($src, $args, $context, $info);

        if( ! $value || $value === "/no_selection") {
            return null;
        }

        return $this->resizeImage($src, camelToSnakeCase($info->fieldName), $value, $args);
    }
}
